package eu.dnetlib.data.mapreduce.hbase.broker.enrich;

import java.io.IOException;
import java.util.Arrays;
import java.util.Map;

import org.apache.commons.collections.MapUtils;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.TableMapper;
import org.apache.hadoop.hbase.util.Bytes;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.protobuf.InvalidProtocolBufferException;

import eu.dnetlib.data.mapreduce.util.DedupUtils;
import eu.dnetlib.data.mapreduce.util.OafDecoder;
import eu.dnetlib.data.mapreduce.util.OafRowKeyDecoder;
import eu.dnetlib.data.proto.FieldTypeProtos.KeyValue;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.OafProtos.OafRel;
import eu.dnetlib.data.proto.TypeProtos.Type;

public class PrepareEnrichementMapper extends TableMapper<ImmutableBytesWritable, ImmutableBytesWritable> {

	@Override
	protected void map(final ImmutableBytesWritable key, final Result value, final Context context) throws IOException, InterruptedException {

		final Type type = getEntityType(key);

		final byte[] body = value.getValue(Bytes.toBytes(type.toString()), Bytes.toBytes("body"));

		if (body == null) {
			context.getCounter("body", "empty oaf").increment(1);
			return;
		}

		switch (type) {
		case project:
			for (final ImmutableBytesWritable resultId : listRelatedIds(value, "resultProject_outcome_produces")) {
				context.write(resultId, new ImmutableBytesWritable(body));
			}
			break;
		case result:
			final Map<byte[], byte[]> mergedIn = value.getFamilyMap(DedupUtils.getDedupCF_mergedInBytes(Type.result));

			if (MapUtils.isNotEmpty(mergedIn)) { // This is to handle the Dedup Based Enrichemnt Events
				context.write(getRootId(mergedIn), new ImmutableBytesWritable(body));
			} else if (!DedupUtils.isRoot(key)) {  // This is to handle the projects related Events
				final Oaf.Builder oafBuilder = Oaf.newBuilder(OafDecoder.decode(body).getOaf());
				for (final String relName : Arrays.asList("resultProject_outcome_isProducedBy")) { //  TODO add dataset rels
					handleRels(oafBuilder, value, body, relName);
				}
				context.write(new ImmutableBytesWritable(key.copyBytes()), new ImmutableBytesWritable(oafBuilder.build().toByteArray()));
			}
			break;
		default:
			context.getCounter("invalid entity", type.toString()).increment(1);
			break;
		}
	}

	public void handleRels(final Oaf.Builder oafBuilder, final Result value, final byte[] body, final String relName) throws IOException, InterruptedException {
		final Iterable<Oaf> rels = listRelations(value, relName);
		if (!Iterables.isEmpty(rels)) {
			for (final Oaf oaf : rels) {
				final String provenance = oaf.getDataInfo().getProvenanceaction().getClassid();
				final OafRel.Builder relBuilder = OafRel.newBuilder(oaf.getRel());

				relBuilder.addCollectedfrom(KeyValue.newBuilder().setKey(provenance));
				oafBuilder.getEntityBuilder().addCachedRel(relBuilder.build());
			}
		}
	}

	private Iterable<Oaf> listRelations(final Result value, final String relType) {
		final Map<byte[], byte[]> map = value.getFamilyMap(Bytes.toBytes(relType));
		if (map != null) {
			return Iterables.transform(map.keySet(), new Function<byte[], Oaf>() {

				@Override
				public Oaf apply(final byte[] input) {
					try {
						return Oaf.parseFrom(input);
					} catch (final InvalidProtocolBufferException e) {
						throw new RuntimeException(e);
					}
				}
			});
		} else {
			return Lists.newArrayList();
		}
	}

	private Iterable<ImmutableBytesWritable> listRelatedIds(final Result value, final String relType) {
		final Map<byte[], byte[]> map = value.getFamilyMap(Bytes.toBytes(relType));
		if (map != null) {
			return Iterables.transform(map.keySet(), new Function<byte[], ImmutableBytesWritable>() {

				@Override
				public ImmutableBytesWritable apply(final byte[] input) {
					return new ImmutableBytesWritable(input);
				}
			});
		} else {
			return Lists.newArrayList();
		}
	}

	private Type getEntityType(final ImmutableBytesWritable key) {
		return OafRowKeyDecoder.decode(key.copyBytes()).getType();
	}

	private ImmutableBytesWritable getRootId(final Map<byte[], byte[]> mergedIn) {
		final Iterable<ImmutableBytesWritable> ids = Iterables.transform(mergedIn.keySet(), new Function<byte[], ImmutableBytesWritable>() {

			@Override
			public ImmutableBytesWritable apply(final byte[] input) {
				return new ImmutableBytesWritable(input);
			}
		});
		try {
			return Iterables.getOnlyElement(ids);
		} catch (final IllegalArgumentException e) {
			System.err.println(ids);
			throw e;
		}
	}

}
