package eu.dnetlib.data.mapreduce.hbase.broker;

import java.io.IOException;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import eu.dnetlib.broker.objects.OpenAireEventPayload;
import eu.dnetlib.data.mapreduce.hbase.broker.mapping.HighlightFactory;
import eu.dnetlib.data.mapreduce.hbase.broker.mapping.OpenAireEventPayloadFactory;
import eu.dnetlib.data.mapreduce.hbase.broker.model.EventMessage;
import eu.dnetlib.data.proto.FieldTypeProtos.StructuredProperty;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Reducer.Context;

import static eu.dnetlib.data.mapreduce.hbase.broker.mapping.EventFactory.asEvent;

/**
 * Created by claudio on 26/07/16.
 */
public class PIDEventFactory {

	protected Text tKey = new Text("");

	public static void process(final Context context, final Oaf current, final Oaf other, final float trust) throws IOException, InterruptedException {
		new PIDEventFactory().processPID(context, current, other, trust);
	}

	public void processPID(final Context context, final Oaf current, final Oaf other, final float trust) throws IOException, InterruptedException {
		//PIDS
		if (current.getEntity().getPidList().isEmpty()) {
			for(StructuredProperty pidOther : other.getEntity().getPidList()) {
				doProcessPID(context, current, other, pidOther, Topic.MISSING_PID, trust);
			}
		}

		for(StructuredProperty pidOther : other.getEntity().getPidList()) {
			if(!hasPidValue(current, pidOther.getValue())) {
				doProcessPID(context, current, other, pidOther, Topic.MORE_PID, trust);
			}
		}
	}

	private void doProcessPID(final Context context, final Oaf current, final Oaf other, final StructuredProperty pidOther, final Topic topic, final float trust)
			throws IOException, InterruptedException {
		final Oaf.Builder prototype = Oaf.newBuilder(current);
		prototype.getEntityBuilder().addPid(pidOther);
		final Oaf oaf = prototype.build();

		final EventMessage event = asEvent(oaf.getEntity(), topic, other.getEntity(), trust);
		final OpenAireEventPayload payload = OpenAireEventPayloadFactory.fromOAF(oaf.getEntity(), other.getEntity(), trust);
		event.setPayload(HighlightFactory.highlightEnrichPid(payload, Lists.newArrayList(pidOther)).toJSON());

		context.write(tKey, new Text(event.toString()));
		context.getCounter("event", topic.getValue()).increment(1);
	}

	private boolean hasPidValue(final Oaf oaf, final String value) {
		return Iterables.any(oaf.getEntity().getPidList(), new Predicate<StructuredProperty>() {
			@Override
			public boolean apply(final StructuredProperty pid) {
				return pid.getValue().equalsIgnoreCase(value) ;
			}
		});
	}
}
