package eu.dnetlib.data.mapreduce.hbase.broker;

import java.io.IOException;
import java.util.List;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import eu.dnetlib.broker.objects.OpenAireEventPayload;
import eu.dnetlib.data.mapreduce.hbase.broker.mapping.HighlightFactory;
import eu.dnetlib.data.mapreduce.hbase.broker.mapping.OpenAireEventPayloadFactory;
import eu.dnetlib.data.mapreduce.hbase.broker.model.EventMessage;
import eu.dnetlib.data.proto.FieldTypeProtos.StringField;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Reducer.Context;

import static eu.dnetlib.data.mapreduce.hbase.broker.mapping.EventFactory.asEvent;

/**
 * Created by claudio on 26/07/16.
 */
public class AbstractEventFactory {

	protected Text tKey = new Text("");

	public static void process(final Context context, final Oaf current, final Oaf other, final float trust) throws IOException, InterruptedException {
		new AbstractEventFactory().processAbstract(context, current, other, trust);
	}

	private void processAbstract(final Context context, final Oaf current, final Oaf other, final float trust) throws IOException, InterruptedException {
		if (hasAbstract(other)) {
			//doProcessAbstract(context, current, other, Topic.MORE_ABSTRACT);

			if (!hasAbstract(current)) {
				doProcessAbstract(context, current, other, Topic.MISSING_ABSTRACT, trust);
			}
		}
	}

	private void doProcessAbstract(final Context context, final Oaf current, final Oaf other, final Topic topic, final float trust) throws IOException, InterruptedException {
		final Oaf.Builder prototype = Oaf.newBuilder(current);
		final List<StringField> descriptionList = other.getEntity().getResult().getMetadata().getDescriptionList();
		prototype.getEntityBuilder().getResultBuilder().getMetadataBuilder().addAllDescription(descriptionList);

		final Oaf oaf = prototype.build();

		final EventMessage event = asEvent(oaf.getEntity(), topic, other.getEntity(), trust);
		final OpenAireEventPayload payload = OpenAireEventPayloadFactory.fromOAF(oaf.getEntity(), other.getEntity(), trust);
		event.setPayload(HighlightFactory.highlightEnrichAbstract(payload, descriptionList).toJSON());

		context.write(tKey, new Text(event.toString()));
		context.getCounter("event", topic.getValue()).increment(1);
	}

	private boolean hasAbstract(final Oaf oaf) {
		return Iterables.any(oaf.getEntity().getResult().getMetadata().getDescriptionList(), new Predicate<StringField>() {
			@Override
			public boolean apply(final StringField s) {
				return StringUtils.isNotBlank(s.getValue());
			}
		});
	}

}
