package eu.dnetlib.functionality.modular.ui.dedup;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.UUID;

import javax.annotation.Resource;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.impl.CloudSolrServer;
import org.apache.solr.common.SolrInputDocument;
import org.dom4j.DocumentException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import eu.dnetlib.data.mapreduce.util.OafDecoder;
import eu.dnetlib.data.mapreduce.util.OafEntityDecoder;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.OafProtos.Oaf.Builder;
import eu.dnetlib.data.proto.OafProtos.OafEntity;
import eu.dnetlib.data.transform.OafEntityMerger;
import eu.dnetlib.data.transform.ProtoDocumentMapper;
import eu.dnetlib.enabling.database.rmi.DatabaseService;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpDocumentNotFoundException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.enabling.resultset.client.ResultSetClientFactory;
import eu.dnetlib.functionality.index.client.IndexClient;
import eu.dnetlib.functionality.index.client.IndexClientException;
import eu.dnetlib.functionality.index.client.ResolvingIndexClientFactory;
import eu.dnetlib.functionality.index.client.response.LookupResponse;
import eu.dnetlib.functionality.index.solr.feed.InputDocumentFactory;
import eu.dnetlib.functionality.modular.ui.AbstractAjaxController;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.pace.config.DedupConfig;

@Controller
public class DedupServiceInternalController extends AbstractAjaxController {

	private static final String ID_PREFIX_REGEX = "^\\d\\d\\|";

	private static final Log log = LogFactory.getLog(DedupServiceInternalController.class);

	@Resource
	private UniqueServiceLocator serviceLocator;

	@Autowired
	private ResultSetClientFactory resultSetClientFactory;

	@Value("${dnet.dedup.db.name}")
	private String dbName;

	/** The index client factory. */
	@Autowired
	private ResolvingIndexClientFactory indexClientFactory;

	@Value("${dnet.dedup.index.format}")
	private String indexFormat;

	@Value("${dnet.dedup.index.collection}")
	private String dedupIndexCollection;

	private IndexClient indexClient = null;

	public class OafResult {

		private long total;

		private List<Map<String, String>> results;

		public OafResult(final long total, final List<Map<String, String>> results) {
			super();
			this.setTotal(total);
			this.setResults(results);
		}

		public long getTotal() {
			return total;
		}

		public void setTotal(final long total) {
			this.total = total;
		}

		public List<Map<String, String>> getResults() {
			return results;
		}

		public void setResults(final List<Map<String, String>> results) {
			this.results = results;
		}
	}

	@ResponseBody
	@RequestMapping(value = "/ui/dedup/lookupConfigurations.do")
	public Map<String, List<String>> lookupConfigurations() throws ISLookUpException {
		final Map<String, List<String>> res = Maps.newHashMap();

		final ISLookUpService lookUpService = serviceLocator.getService(ISLookUpService.class);
		final String listEntityTypesXQuery =
				"distinct-values(for $x in //RESOURCE_PROFILE[.//RESOURCE_TYPE/@value = 'DedupOrchestrationDSResourceType'] return $x//ENTITY/@name/string())";

		for (final String entityType : lookUpService.quickSearchProfile(listEntityTypesXQuery)) {
			final String xquery =
					String.format(
							"for $x in //RESOURCE_PROFILE[" +
									".//RESOURCE_TYPE/@value = 'DedupOrchestrationDSResourceType' and .//ENTITY/@name='%s' ] " +
									"return $x//ACTION_SET/@id/string()", entityType);
			res.put(entityType, lookUpService.quickSearchProfile(xquery));
		}
		return res;
	}

	@ResponseBody
	@RequestMapping(value = "/ui/dedup/search.do")
	public OafResult search(@RequestParam(value = "entityType", required = true) final String type,
			@RequestParam(value = "query", required = true) final String userQuery,
			@RequestParam(value = "actionSet", required = true) final String actionSet,
			@RequestParam(value = "start", required = true) final int start,
			@RequestParam(value = "rows", required = true) final int rows,
			@RequestParam(value = "fields", required = true) final String fields) throws Exception {

		try {
			final String cqlQuery =
					String.format("(>s=SOLR s.q.op=AND) and oaftype = %s and actionset exact \"%s\" and deletedbyinference = false and %s", type, actionSet,
							userQuery);

			final LookupResponse rsp = getIndexClient().lookup(cqlQuery, null, start, (start + rows) - 1);

			final List<String> fieldList = Lists.newLinkedList(Splitter.on(",").omitEmptyStrings().trimResults().split(fields));
			final List<Map<String, String>> resList = Lists.newLinkedList(Iterables.transform(toOaf(rsp), getOaf2FieldMapFunction(type, fieldList)));

			return new OafResult(rsp.getTotal(), resList);
		} catch (final Exception e) {
			log.error("search error", e);
			throw e;
		}
	}

	@ResponseBody
	@RequestMapping(value = "/ui/dedup/searchById.do")
	public OafResult searchById(@RequestParam(value = "entityType", required = true) final String type,
			@RequestParam(value = "objidentifier", required = true) final String objidentifier,
			@RequestParam(value = "fields", required = true) final List<String> fields) throws Exception {

		final String cqlQuery = "objidentifier exact \"" + objidentifier + "\"";

		final LookupResponse rsp = getIndexClient().lookup(cqlQuery, null, 0, 1);

		final Iterable<Oaf> oafList = toOaf(rsp);

		final List<Map<String, String>> resList = Lists.newLinkedList(Iterables.transform(oafList, getOaf2FieldMapFunction(type, fields)));

		return new OafResult(rsp.getTotal(), resList);
	}

	@ResponseBody
	@RequestMapping(value = "/ui/dedup/commit.do")
	public boolean addSimRels(@RequestBody(required = true) final SimilarityGroup group) throws Exception {
		try {
			final DatabaseService dbService = serviceLocator.getService(DatabaseService.class);
			final String version = InputDocumentFactory.getParsedDateField(DateUtils.now_ISO8601());

			group.setId(UUID.randomUUID().toString());
			group.setDate(version);

			// relational DB update
			log.info("adding similarities: " + group.getGroup());
			updateGroupSql(dbService, group, version);

			log.info("adding dissimilarities: " + group.getDissimilar());
			dissimilaritiesSql(dbService, group);

			// index update
			log.info("starting index update");

			final CloudSolrServer solrServer = getSolrServer();
			final ProtoDocumentMapper mapper = initProtoMapper();

			final Map<String, String> config = Maps.newHashMap();
			config.put("entityType", group.getEntityType().getType());
			config.put("configurationId", group.getActionSet());
			final DedupConfig dedupConf = DedupConfig.loadDefault(config);

			for (final String rootId : group.getRootIds()) {
				solrServer.deleteById(rootId);
			}
			final Function<Oaf, SolrInputDocument> oaf2solr = oaf2solr(version, group.getActionSet(), mapper);
			final List<SolrInputDocument> buffer = Lists.newLinkedList();

			final List<Oaf> groupDocs = Lists.newArrayList(markDeleted(asOafBuilder(parseBase64(queryIndex(group.getGroup())))));

			buffer.addAll(Lists.newArrayList(asIndexDocs(oaf2solr, groupDocs)));
			final SolrInputDocument newRoot = oaf2solr.apply(OafEntityMerger.merge(dedupConf, newRootId(group), groupDocs).build());
			newRoot.setField("actionset", dedupConf.getWf().getConfigurationId());
			buffer.add(newRoot);

			final List<Oaf> dissimDocs = Lists.newArrayList(markUnDeleted(asOafBuilder(parseBase64(queryIndex(group.getDissimilar().keySet())))));
			buffer.addAll(Lists.newArrayList(asIndexDocs(oaf2solr, dissimDocs)));

			log.debug(String.format("adding %d documents to index %s", buffer.size(), dedupIndexCollection));

			final int addStatus = solrServer.add(buffer).getStatus();
			log.debug("solr add status: " + addStatus);

			final int commitStatus = solrServer.commit().getStatus();
			log.debug("solr commit status: " + commitStatus);

			return (addStatus == 0) && (commitStatus == 0);
		} catch (final Exception e) {
			log.error(e);
			throw e;
		}
	}

	// helpers

	private IndexClient getIndexClient() throws IndexClientException, ISLookUpDocumentNotFoundException, ISLookUpException {
		if (indexClient == null) {
			indexClient = indexClientFactory.getClient(indexFormat, "index", "dedup", "solr");
		}
		return indexClient;
	}

	private static Map<String, Map<String, String>> paths = Maps.newHashMap();

	static {
		paths.put("result", new HashMap<String, String>());
		paths.put("organization", new HashMap<String, String>());
		paths.put("person", new HashMap<String, String>());

		paths.get("result").put("title", "result/metadata/title/value");
		paths.get("result").put("dateofacceptance", "result/metadata/dateofacceptance/value");
		paths.get("result").put("description", "result/metadata/description/value");
		paths.get("result").put("author", "result/author/metadata/fullname/value");

		paths.get("organization").put("legalname", "organization/metadata/legalname/value");
		paths.get("organization").put("legalshortname", "organization/metadata/legalshortname/value");
		paths.get("organization").put("websiteurl", "organization/metadata/websiteurl/value");
		paths.get("organization").put("country", "organization/metadata/country/classid");

		paths.get("person").put("fullname", "person/metadata/fullname/value");
	}

	private Function<Oaf, Map<String, String>> getOaf2FieldMapFunction(final String type, final List<String> fields) {
		return new Function<Oaf, Map<String, String>>() {

			@Override
			public Map<String, String> apply(final Oaf oaf) {

				final OafEntityDecoder ed = OafDecoder.decode(oaf).decodeEntity();
				final Map<String, String> res = Maps.newHashMap();
				final String oafId = cleanId(oaf.getEntity().getId());
				final List<String> idList = Lists.newArrayList(Iterables.transform(oaf.getEntity().getChildrenList(), new Function<OafEntity, String>() {

					@Override
					public String apply(final OafEntity e) {
						return cleanId(e.getId());
					}
				}));
				if (idList.isEmpty()) {
					idList.add(oafId);
				}
				res.put("id", oafId);
				res.put("idList", Joiner.on(",").join(idList));
				res.put("groupSize", idList.isEmpty() ? "1" : idList.size() + "");

				for (final String fieldName : fields) {
					res.put(fieldName, Joiner.on("; ").skipNulls().join(ed.getFieldValues(fieldName, paths.get(type).get(fieldName))));
				}

				return res;
			}
		};
	}

	private String cleanId(final String id) {
		return id.replaceFirst(ID_PREFIX_REGEX, "");
	}

	private String newRootId(final SimilarityGroup group) {
		if (group.getRootIds().isEmpty()) return "dedup_wf_001::" + Collections.min(group.getGroup()).replaceFirst("^.*::", "");
		else return Collections.min(group.getRootIds());
	}

	private Iterable<SolrInputDocument> asIndexDocs(final Function<Oaf, SolrInputDocument> mapper, final Iterable<Oaf> a) {
		return Iterables.transform(a, mapper);
	}

	private Function<Oaf, SolrInputDocument> oaf2solr(final String version, final String actionSetId, final ProtoDocumentMapper mapper) {
		return new Function<Oaf, SolrInputDocument>() {

			@Override
			public SolrInputDocument apply(final Oaf oaf) {
				try {
					return mapper.map(oaf, version, "", actionSetId);
				} catch (final Throwable e) {
					throw new IllegalArgumentException("unable to map proto to index document", e);
				}
			}
		};
	}

	private Iterable<String> queryIndex(final Iterable<String> ids) {
		return Iterables.transform(ids, idToIndexDocumentMapper());
	}

	private Function<String, String> idToIndexDocumentMapper() {
		return new Function<String, String>() {

			@Override
			public String apply(final String id) {
				try {
					final String cql = "objidentifier exact \"" + id + "\"";
					final LookupResponse rsp = getIndexClient().lookup(cql, null, 0, 1);

					log.debug(String.format("query index for id '%s', found '%d'", id, rsp.getTotal()));

					return Iterables.getOnlyElement(rsp.getRecords());
				} catch (final Throwable e) {
					throw new RuntimeException("unable to query id: " + id);
				}
			}
		};
	}

	private Iterable<Oaf> parseBase64(final Iterable<String> r) {
		return Iterables.transform(r, getXml2OafFunction());
	}

	private ProtoDocumentMapper initProtoMapper() throws DocumentException, ISLookUpException, ISLookUpDocumentNotFoundException {
		return new ProtoDocumentMapper(
				serviceLocator
						.getService(ISLookUpService.class)
						.getResourceProfileByQuery(
								"collection('')//RESOURCE_PROFILE[.//RESOURCE_TYPE/@value = 'MDFormatDSResourceType' and .//NAME='" + indexFormat
										+ "']//LAYOUT[@name='index']/FIELDS"));
	}

	private Iterable<Oaf> markDeleted(final Iterable<Oaf.Builder> builders) {
		return Iterables.transform(builders, new Function<Oaf.Builder, Oaf>() {

			@Override
			public Oaf apply(final Oaf.Builder builder) {
				// TODO add more changes to the Oaf object here as needed.
				builder.getDataInfoBuilder().setDeletedbyinference(true);
				return builder.build();
			}
		});
	}

	private Iterable<Oaf> markUnDeleted(final Iterable<Oaf.Builder> builders) {
		return Iterables.transform(builders, new Function<Oaf.Builder, Oaf>() {

			@Override
			public Oaf apply(final Oaf.Builder builder) {
				// TODO add more changes to the Oaf object here as needed.
				builder.getDataInfoBuilder().setDeletedbyinference(false);
				return builder.build();
			}
		});
	}

	private Iterable<Oaf.Builder> asOafBuilder(final Iterable<Oaf> oaf) {
		return Iterables.transform(oaf, new Function<Oaf, Oaf.Builder>() {

			@Override
			public Builder apply(final Oaf oaf) {
				return Oaf.newBuilder(oaf);
			}
		});
	}

	private Function<String, Oaf> getXml2OafFunction() {
		return new Function<String, Oaf>() {

			@Override
			public Oaf apply(final String s) {
				// final String base64 = s.replaceAll("<record.*>", "").replace("</record>", "");
				final String base64 = StringUtils.substringBefore(StringUtils.substringAfter(s, ">"), "<");
				try {
					final byte[] oaf = Base64.decodeBase64(base64);
					return OafDecoder.decode(oaf).getOaf();
				} catch (final Throwable e) {
					throw new IllegalArgumentException("unable to decode base64 encoded Oaf object: " + base64);
				}
			}
		};
	}

	private Iterable<Oaf> toOaf(final LookupResponse rsp) {
		return Iterables.transform(rsp.getRecords(), getXml2OafFunction());
	}

	private void updateGroupSql(final DatabaseService dbService, final SimilarityGroup group, final String version) throws Exception {
		for (final String id : group.getGroup()) {
			// sql.append(String.format("DELETE FROM entities WHERE objidentifier = '%s'; ", id));
			safeUpdateSql(dbService, String.format("DELETE FROM similarity_groups WHERE objidentifier = '%s'; ", id));
		}
		final String type = group.getEntityType().getType();
		safeUpdateSql(dbService,
				String.format("INSERT INTO groups(id, entitytype, date, actionsetid) VALUES('%s', '%s', '%s', '%s'); ", group.getId(), type, version,
						group.getActionSet()));
		for (final String id : group.getGroup()) {
			if (!dbService.contains(dbName, "entities", "id", id)) {
				safeUpdateSql(dbService, String.format("INSERT INTO entities(id, entitytype) VALUES('%s', '%s'); ", id, type));
			}

			// throw new Exception("id already defined in a similarity group.");
			safeUpdateSql(dbService, String.format("INSERT INTO similarity_groups(groupid, objidentifier) VALUES('%s', '%s'); ", group.getId(), id));
		}
	}

	private void dissimilaritiesSql(final DatabaseService dbService, final SimilarityGroup group) {

		final String type = group.getEntityType().getType();

		for (final Entry<String, Set<String>> e : group.getDissimilar().entrySet()) {
			if (!dbService.contains(dbName, "entities", "id", e.getKey())) {
				safeUpdateSql(dbService, String.format("INSERT INTO entities(id, entitytype) VALUES('%s', '%s'); ", e.getKey(), type));
			}
			for (final String id : e.getValue()) {
				if (!dbService.contains(dbName, "entities", "id", id)) {
					safeUpdateSql(dbService, String.format("INSERT INTO entities(id, entitytype) VALUES('%s', '%s'); ", id, type));
				}
			}
		}

		for (final Entry<String, Set<String>> e : group.getDissimilar().entrySet()) {
			for (final String id : e.getValue()) {
				safeUpdateSql(dbService,
						String.format("INSERT INTO dissimilarities(id1, id2, actionsetid) VALUES('%s', '%s', '%s'); ", e.getKey(), id, group.getActionSet()));
			}
		}
	}

	private boolean safeUpdateSql(final DatabaseService dbService, final String sql) {
		try {
			log.info(sql);
			return dbService.updateSQL(dbName, sql);
		} catch (final Throwable e) {
			log.error(e.getMessage());
			log.debug(ExceptionUtils.getFullStackTrace(e));
			return false;
		}
	}

	private CloudSolrServer getSolrServer() {
		final String zk = getIndexSolrUrlZk();
		log.info(String.format("initializing solr client for collection %s, zk url: %s", dedupIndexCollection, zk));
		final CloudSolrServer solrServer = new CloudSolrServer(zk);
		solrServer.setDefaultCollection(dedupIndexCollection);

		return solrServer;
	}

	private String getIndexSolrUrlZk() {
		try {
			return getResourceProfileByQuery("for $x in /RESOURCE_PROFILE[.//RESOURCE_TYPE/@value='IndexServiceResourceType'] return $x//PROTOCOL[./@name='solr']/@address/string()");
		} catch (final ISLookUpException e) {
			throw new IllegalStateException("unable to read solr ZK url from service profile", e);
		}
	}

	private String getResourceProfileByQuery(final String xquery) throws ISLookUpException {
		log.debug("quering for service property: " + xquery);
		final String res = serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(xquery);
		if (StringUtils.isBlank(res)) throw new IllegalStateException("unable to find unique service property, xquery: " + xquery);
		return res;
	}

}
