package eu.dnetlib.actionmanager.hbase;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import javax.annotation.Resource;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import eu.dnetlib.actionmanager.is.ISClient;
import eu.dnetlib.actionmanager.rmi.ActionManagerException;
import eu.dnetlib.actionmanager.set.ActionManagerSet;
import eu.dnetlib.data.hadoop.config.ClusterName;
import eu.dnetlib.data.hadoop.rmi.HadoopBlackboardActions;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.enabling.tools.blackboard.BlackboardClientHandler;
import eu.dnetlib.enabling.tools.blackboard.BlackboardJob;
import eu.dnetlib.enabling.tools.blackboard.BlackboardJobRegistry;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.beans.factory.annotation.Value;

public class HBaseJobLauncher {

	public static final String ALL_SETS = "__ALL__";
	public static final String SEMICOLON = ";";
	public static final String COLON = ",";
	public static final String SEQFILE_INPUTFORMAT = "org.apache.hadoop.mapreduce.lib.input.SequenceFileInputFormat";
	private static final Log log = LogFactory.getLog(HBaseJobLauncher.class); // NOPMD by marko on 11/24/08 5:02 PM
	private final Executor executor = Executors.newCachedThreadPool();

	@Resource
	private UniqueServiceLocator serviceLocator;

	private ISClient infomationServiceClient;

	/**
	 * blackboard handler.
	 */
	@Resource
	private BlackboardClientHandler blackboardClientHandler;

	/**
	 * blackboard job registry.
	 */
	@Resource
	private BlackboardJobRegistry jobRegistry;

	@Value("${services.actionmanager.promote.mapper.class}")
	private String promoteJobMapperClass;

	private String actionTable;
	private String dataTable;

	public void executeGarbageJob(final HBaseCallback callback) throws ActionManagerException {
		log.info("Starting clean garbage m/r job");

		final Set<String> rawSets = infomationServiceClient.listValidRawSets();

		final Map<String, String> properties = Maps.newHashMap();
		properties.put("garbageTimeMargin", infomationServiceClient.getGarbageTimeMargin());
		executeHBASE("garbageActionsJob", rawSets, actionTable, actionTable, properties, callback);
	}

	@Deprecated
	public void executePromoteJob(final Set<String> sets, final HBaseCallback callback) throws ActionManagerException {
		log.info("Starting commit m/r job; sets=" + sets);

		final Set<String> rawSets = Sets.newHashSet();
		if (sets == null || sets.isEmpty()) {
			rawSets.addAll(infomationServiceClient.listValidRawSets());
		} else {
			for (String set : sets) {
				if (infomationServiceClient.existsSet(set)) {
					rawSets.add(infomationServiceClient.geLatestRawSet(set).getId());
				} else {
					log.error("Invalid set " + set);
					throw new ActionManagerException("Invalid set " + set);
				}
			}
		}

		executeHBASE("promoteActionsJob", rawSets, actionTable, dataTable, new HashMap<String, String>(), callback);
	}

	public void executePromoteFromHDFSJob(final Set<String> sets, final HBaseCallback callback) throws ActionManagerException, ISLookUpException {
		log.info("Starting commit m/r job; sets=" + sets);

		final List<String> paths = Lists.newArrayList();
		if (sets == null || sets.isEmpty()) {

			for (ActionManagerSet set : infomationServiceClient.listSets()) {
				paths.add(set.getPathToLatest());
			}

		} else {
			for (String setId : sets) {
				if (infomationServiceClient.existsSet(setId)) {
					final ActionManagerSet set = infomationServiceClient.getSet(setId);
					paths.add(set.getPathToLatest());
				} else {
					log.error("Invalid set " + setId);
					throw new ActionManagerException("Invalid set " + setId);
				}
			}
		}

		final Map<String, String> params = Maps.newHashMap();
		params.put("mapred.input.dir.formats", Joiner.on(COLON).join(Iterables.transform(paths, new Function<String, String>() {
			@Override
			public String apply(final String path) {
				return path + SEMICOLON + SEQFILE_INPUTFORMAT;
			}
		})));
		params.put("mapred.input.dir.mappers", Joiner.on(COLON).join(Iterables.transform(paths, new Function<String, String>() {
			@Override
			public String apply(final String path) {
				return path + SEMICOLON + promoteJobMapperClass;
			}
		})));

		params.put("hbase.mapred.outputtable", dataTable);
		params.put("hbase.mapreduce.outputtable", dataTable);

		log.info("promoting HDFS rawsets: " + paths);

		executeHDFS("promoteMultipleActionSetsJob", params, callback);
	}

	private void executeHDFS(final String jobName, final Map<String, String> params, final HBaseCallback callback) throws ActionManagerException {

		if (params == null || params.isEmpty()) { throw new ActionManagerException("Missing HDFS paths"); }

		executor.execute(new Runnable() {

			@Override
			public void run() {
				try {
					final String serviceId = findHadoopServiceProfileID();
					final BlackboardJob bbJob = blackboardClientHandler.newJob(serviceId);

					bbJob.setAction(HadoopBlackboardActions.SUBMIT_MAPREDUCE_JOB.toString());
					bbJob.getParameters().put("job.name", jobName);
					bbJob.getParameters().put("cluster", ClusterName.DM.toString());
					bbJob.getParameters().putAll(params);

					jobRegistry.registerJobListener(bbJob, new HBaseBlackboardJobListener(callback));
					blackboardClientHandler.assign(bbJob);
				} catch (Exception e) {
					log.error("Error launching m/r job: " + jobName, e);
					throw new RuntimeException("Error launching m/r job: " + jobName, e);
				}
			}

		});
	}

	@Deprecated
	private void executeHBASE(final String jobName,
			final Set<String> rawSets,
			final String inputTable,
			final String outputTable,
			final Map<String, String> properties,
			final HBaseCallback callback)
			throws ActionManagerException {

		if (rawSets == null || rawSets.isEmpty()) { throw new ActionManagerException("Missing rawsets"); }

		executor.execute(new Runnable() {

			@Override
			public void run() {
				try {
					final String serviceId = findHadoopServiceProfileID();
					final BlackboardJob bbJob = blackboardClientHandler.newJob(serviceId);

					final String rawsets = Joiner.on(COLON).skipNulls().join(rawSets);
					log.info("promoting rawsets: " + rawsets);

					bbJob.setAction(HadoopBlackboardActions.SUBMIT_MAPREDUCE_JOB.toString());
					bbJob.getParameters().put("job.name", jobName);
					bbJob.getParameters().put("cluster", ClusterName.DM.toString());
					bbJob.getParameters().put("latestRawSets", rawsets);
					bbJob.getParameters().put("hbase.mapred.inputtable", inputTable);
					bbJob.getParameters().put("hbase.mapreduce.inputtable", inputTable);
					bbJob.getParameters().put("hbase.mapred.outputtable", outputTable);
					bbJob.getParameters().put("hbase.mapreduce.outputtable", outputTable);
					bbJob.getParameters().putAll(properties);

					jobRegistry.registerJobListener(bbJob, new HBaseBlackboardJobListener(callback));
					blackboardClientHandler.assign(bbJob);
				} catch (Exception e) {
					log.error("Error launching m/r job: " + jobName, e);
					throw new RuntimeException("Error launching m/r job: " + jobName, e);
				}
			}

		});
	}

	private String findHadoopServiceProfileID() throws Exception {
		return serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(
				"collection('/db/DRIVER/ServiceResources/HadoopServiceResourceType')//RESOURCE_IDENTIFIER/@value/string()");
	}

	public String getActionTable() {
		return actionTable;
	}

	@Required
	public void setActionTable(final String actionTable) {
		this.actionTable = actionTable;
	}

	public String getDataTable() {
		return dataTable;
	}

	@Required
	public void setDataTable(final String dataTable) {
		this.dataTable = dataTable;
	}

	public ISClient getInfomationServiceClient() {
		return infomationServiceClient;
	}

	@Required
	public void setInfomationServiceClient(final ISClient infomationServiceClient) {
		this.infomationServiceClient = infomationServiceClient;
	}

}
