package eu.dnetlib.openaire.exporter;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.Map;
import java.util.zip.GZIPOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import javax.servlet.ServletOutputStream;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.google.common.collect.Maps;
import com.google.common.xml.XmlEscapers;
import org.antlr.stringtemplate.StringTemplate;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

@Controller
public class ProjectsController {

	private static final Log log = LogFactory.getLog(ProjectsController.class); // NOPMD by marko on 11/24/08 5:02 PM

	@Value("${openaire.exporter.projectsfundings.sql.template}")
	private Resource projectsFundingQueryTemplate;

	@Value("${openaire.exporter.projects2tsv.sql.template}")
	private Resource projects2tsvQueryTemplate;

	@Value("${openaire.exporter.dspace.template.project}")
	private Resource dspaceTemplate;

	@Value("${openaire.exporter.dspace.template.head}")
	private Resource dspaceHeadTemplate;

	@Value("${openaire.exporter.dspace.template.tail}")
	private Resource dspaceTailTemplate;

	@Value("${openaire.exporter.eprints.template}")
	private Resource eprintsTemplate;

	@Value("${openaire.exporter.projectdetails.sql}")
	private Resource projectDetailsSql;

	@Autowired
	private JdbcApiDao dao;

	@Autowired
	private ProjectQueryParamsFactory projectQueryParamsFactory;

	public enum Funding {
		FP7, WT, FCT, H2020, NHMRC, ARC, SFI, MZOS, HRZZ, NWO, MESTD, FWF
	}

	@RequestMapping(value = "/openaire/export/**/project/dspace.do")
	void processDspace(final HttpServletRequest request,
			final ServletResponse response,
			@RequestParam(value = "startFrom", required = false) final String startFrom,
			@RequestParam(value = "startUntil", required = false) final String startUntil,
			@RequestParam(value = "endFrom", required = false) final String endFrom,
			@RequestParam(value = "endUntil", required = false) final String endUntil) throws Exception {


		final ProjectQueryParams params = projectQueryParamsFactory.generateParams(request, startFrom, startUntil, endFrom, endUntil);

		final StringTemplate headSt = new StringTemplate(IOUtils.toString(dspaceHeadTemplate.getInputStream()));
		headSt.setAttribute("fundingProgramme", params.getFundingProgramme());

		final StringTemplate tailSt = new StringTemplate(IOUtils.toString(dspaceTailTemplate.getInputStream()));

		response.setContentType("text/xml");
		doProcess(response, params, headSt.toString(), dspaceTemplate, tailSt.toString(), new ValueCleaner() {
			@Override
			public String clean(final String s) {
				return XmlEscapers.xmlContentEscaper().escape(oneLiner(s));
			}
		});
	}

	@RequestMapping(value = "/openaire/export/**/project/eprints.do")
	void processEprints(final HttpServletRequest request,
			final ServletResponse response,
			@RequestParam(value = "startFrom", required = false) final String startFrom,
			@RequestParam(value = "startUntil", required = false) final String startUntil,
			@RequestParam(value = "endFrom", required = false) final String endFrom,
			@RequestParam(value = "endUntil", required = false) final String endUntil) throws Exception {

		final ProjectQueryParams params = projectQueryParamsFactory.generateParams(request, startFrom, startUntil, endFrom, endUntil);
		response.setContentType("text/html");
		doProcess(response, params, null, eprintsTemplate, null, new ValueCleaner() {
			@Override
			public String clean(final String s) {
				return oneLiner(s);
			}
		});
	}

	private String oneLiner(final String s) {
		return StringUtils.isNotBlank(s) ? s.replaceAll("\\n", " ").trim() : "";
	}

	private void doProcess(
			final ServletResponse response,
			final ProjectQueryParams params,
			final String head, final Resource projectTemplate, final String tail,
			final ValueCleaner cleaner) throws IOException, SQLException {

		final StringTemplate st = new StringTemplate(IOUtils.toString(projectTemplate.getInputStream()));
		dao.streamProjects(obtainQuery(params), response.getOutputStream(), head, st, tail, cleaner);
	}

	@RequestMapping(value = "/openaire/export/project2tsv.do")
	void processTsv(final HttpServletRequest request, final HttpServletResponse response,
			@RequestParam(value = "funding", required = true) final String funding,
			@RequestParam(value = "article293", required = false) final Boolean article293) throws Exception {

		final String date = new SimpleDateFormat("yyyyMMdd").format(new Date());
		final String filename = "projects_" + funding + "_" + date + ".tsv";
		response.setContentType("text/tab-separated-values");
		response.setHeader("Content-Disposition", "attachment; filename=\"" + filename + ".zip\"");

		final StringTemplate st = new StringTemplate(IOUtils.toString(projects2tsvQueryTemplate.getInputStream()));
		Funding requestedFunding = Funding.valueOf(funding.toUpperCase());
		String fundingPrefix = getFundingPrefix(requestedFunding, null);
		log.debug("Setting fundingprefix to " + fundingPrefix);
		st.setAttribute("fundingprefix", fundingPrefix);
		st.setAttribute("filters", expandFilters(article293));

		final ZipOutputStream out = new ZipOutputStream(new BufferedOutputStream(response.getOutputStream()));
		out.putNextEntry(new ZipEntry(filename));
		dao.streamProjectsTSV(st.toString(), out);
	}

	@RequestMapping(value = "/openaire/export/streamProjectDetails.do")
	void streamProjectDetails(final HttpServletResponse response,
			@RequestParam(value = "format", required = true) final String format,
			@RequestParam(value = "compress", required = false) final Boolean compress) throws IOException, SQLException {
		final String sql = IOUtils.toString(projectDetailsSql.getInputStream());

		if (compress != null && compress) {
			response.setHeader("Content-Encoding", "gzip");
		}
		switch (format) {
		case "csv":
			response.setContentType("text/csv");
			break;
		case "json":
			response.setContentType("text/plain");
			break;
		default: throw new IllegalArgumentException("unsupported format: " + format);
		}

		final OutputStream outputStream = getOutputStream(response.getOutputStream(), compress);
		dao.streamProjectDetails(sql, outputStream, format);
	}

	private OutputStream getOutputStream(final ServletOutputStream outputStream, final Boolean compress) throws IOException {
		if (compress != null && compress) {
			return new GZIPOutputStream(outputStream);
		}
		return outputStream;
	}

	@ExceptionHandler({Exception.class, Throwable.class})
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	public @ResponseBody ErrorMessage handleSqlException(final Exception e) {
		log.error(e.getMessage());
		return new ErrorMessage(e);
	}

	public class ErrorMessage {

		private final String message;
		private final String stacktrace;

		public ErrorMessage(final Exception e) {
			this(e.getMessage(), ExceptionUtils.getStackTrace(e));
		}

		public ErrorMessage(final String message, final String stacktrace) {
			this.message = message;
			this.stacktrace = stacktrace;
		}

		public String getMessage() {
			return this.message;
		}

		public String getStacktrace() {
			return this.stacktrace;
		}
	}


	private Map<String, String> expandFilters(final Boolean article293) {
		final Map<String, String> res = Maps.newHashMap();

		if (article293 != null) {
			res.put("ec_article29_3", String.valueOf(article293));
		}

		return res;
	}

	/**
	 * Creates the query on the fundingProgramme specified in the given parameters.
	 *
	 * @param params
	 *            request parameters
	 * @return the query string
	 * @throws IllegalArgumentException
	 *             if the funding program is not recognized
	 * @throws IOException
	 *             if there are problem loading the query temlate
	 * @throws IllegalArgumentException
	 *             if the funding program is not recognized
	 */
	protected String obtainQuery(final ProjectQueryParams params) throws IllegalArgumentException, IOException {
		String funding = params.getFundingProgramme();
		String suffix = params.getFundingPath();
		String fundingPrefix = getFundingPrefix(Funding.valueOf(funding.toUpperCase()), suffix);
		final StringTemplate st = new StringTemplate(IOUtils.toString(projectsFundingQueryTemplate.getInputStream()));
		st.setAttribute("fundingprefix", fundingPrefix);
		String theQuery = setDateParameters(st.toString(), params);
		log.debug("Generated query: " + theQuery);
		return theQuery;
	}

	private String getFundingPrefix(final Funding funding, final String suffix) throws IllegalArgumentException {
		switch (funding) {
		case FCT:
			if (StringUtils.isBlank(suffix)) return "fct_________::FCT::";
			else return "fct_________::FCT::" + suffix;
		case WT:
			if (StringUtils.isBlank(suffix)) return "wt__________::WT::";
			else return "wt__________::WT::" + suffix;
		case FP7:
			if (StringUtils.isBlank(suffix)) return "ec__________::EC::FP7::";
			else return "ec__________::EC::FP7::" + suffix;
		case H2020:
			if (StringUtils.isBlank(suffix)) return "ec__________::EC::H2020";
			else return "ec__________::EC::H2020" + suffix;
		case NHMRC:
			if (StringUtils.isBlank(suffix)) return "nhmrc_______::NHMRC::";
			else return "nhmrc_______::NHMRC::" + suffix;
		case ARC:
			if (StringUtils.isBlank(suffix)) return "arc_________::ARC::";
			else return "arc_________::ARC::" + suffix;
		case SFI:
			if (StringUtils.isBlank(suffix)) return "sfi_________::SFI::";
			else return "sfi_________::SFI::" + suffix;
		case MZOS:
			if (StringUtils.isBlank(suffix)) return "irb_hr______::MZOS";
			else return "irb_hr______::MZOS::" + suffix;
		case HRZZ:
			if (StringUtils.isBlank(suffix)) return "irb_hr______::HRZZ";
			else return "irb_hr______::HRZZ::" + suffix;
		case NWO:
			if (StringUtils.isBlank(suffix)) return "nwo_________::NWO";
			else return "nwo_________::NWO::" + suffix;
		case MESTD:
			if (StringUtils.isBlank(suffix)) return "mestd_______::MESTD";
			else return "mestd_______::MESTD::" + suffix;
		case FWF:
			if (StringUtils.isBlank(suffix)) return "fwf_________::FWF";
			else return "fwf_________::FWF::" + suffix;
		default:
			throw new IllegalArgumentException("Invalid funding " + funding + " (valid are: " + Arrays.asList(Funding.values()) + ") ");
		}
	}

	private String setDateParameters(final String query, final ProjectQueryParams params) {
		String queryWithDates = query;
		if (params.getStartFrom() != null) {
			queryWithDates += " AND startdate >= '" + params.getStartFrom() + "'";
		}
		if (params.getStartUntil() != null) {
			queryWithDates += " AND startdate <= '" + params.getStartUntil() + "'";
		}
		if (params.getEndFrom() != null) {
			queryWithDates += " AND enddate >= '" + params.getEndFrom() + "'";
		}
		if (params.getEndUntil() != null) {
			queryWithDates += " AND enddate <= '" + params.getEndUntil() + "'";
		}
		return queryWithDates;
	}

	public void setProjectsFundingQueryTemplate(final Resource projectsFundingQueryTemplate) {
		this.projectsFundingQueryTemplate = projectsFundingQueryTemplate;
	}

}
