package eu.dnetlib.openaire.exporter;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.sql.*;
import java.util.Arrays;
import java.util.List;
import java.util.zip.GZIPOutputStream;
import java.util.zip.ZipOutputStream;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import eu.dnetlib.miscutils.datetime.HumanTime;
import eu.dnetlib.openaire.exporter.model.Project;
import eu.dnetlib.openaire.exporter.model.ProjectDetail;
import org.antlr.stringtemplate.StringTemplate;
import org.apache.commons.dbcp2.BasicDataSource;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

/**
 * Created by claudio on 20/09/16.
 */
public class JdbcApiDao {

	public static final Charset UTF8 = Charset.forName("UTF-8");
	private static final Log log = LogFactory.getLog(JdbcApiDao.class);
	@Value("${openaire.exporter.jdbc.maxrows}")
	private int maxRows;

	@Value("${openaire.exporter.projectdetails.flushsize}")
	private int gzipFlushSize;

	@Value("${openaire.exporter.projects2tsv.fields}")
	private String tsvFields;

	@Autowired
	private BasicDataSource apiDataSource;

	public void streamProjects(final String sql, final OutputStream out,
			final String head, final StringTemplate projectTemplate, final String tail,
			final ValueCleaner cleaner) throws IOException, SQLException {

		if (log.isDebugEnabled()) {
			log.debug("Thread " + Thread.currentThread().getId() + " begin");
		}
		final long start = System.currentTimeMillis();

		if (StringUtils.isNotBlank(head)) {
			out.write(head.getBytes(UTF8));
		}

		try(final Connection con = getConn(); final PreparedStatement stm = getStm(sql, con); final ResultSet rs = stm.executeQuery()) {
			while(rs.next()){
				final Project p = new Project()
						.setFunder(cleaner.clean(rs.getString("funder")))
						.setJurisdiction(cleaner.clean(rs.getString("jurisdiction")))
						.setFundingpathid(cleaner.clean(rs.getString("fundingpathid")))
						.setAcronym(cleaner.clean(rs.getString("Project Acronym")))
						.setTitle(cleaner.clean(rs.getString("Project Title")))
						.setCode(cleaner.clean(rs.getString("Grant Agreement Number")))
						.setStartdate(cleaner.clean(rs.getString("Start Date")))
						.setEnddate(cleaner.clean(rs.getString("End Date")));

				projectTemplate.reset();
				projectTemplate.setAttribute("p", p);
				out.write(projectTemplate.toString().getBytes(UTF8));
			}
			if (StringUtils.isNotBlank(tail)) {
				out.write(tail.getBytes(UTF8));
			}
			if (log.isDebugEnabled()) {
				log.debug("Thread " + Thread.currentThread().getId() + " ends, took: " + HumanTime.exactly(System.currentTimeMillis() - start));
			}
		} finally {
			out.close();
		}
	}

	public void streamProjectsTSV(final String sql, final ZipOutputStream out) throws IOException, SQLException {

		if (log.isDebugEnabled()) {
			log.debug("Thread " + Thread.currentThread().getId() + " begin");
		}
		final long start = System.currentTimeMillis();
		final List<String> fields = Lists.newArrayList(Splitter.on(",").omitEmptyStrings().trimResults().split(tsvFields));
		writeCSVLine(out, fields);

		try(final Connection con = getConn(); final PreparedStatement stm = getStm(sql, con); final ResultSet rs = stm.executeQuery()) {
			while(rs.next()) {
				final Project p = new Project()
						.setCode(rs.getString("Grant Agreement Number"))
						.setAcronym(rs.getString("Project Acronym"))
						.setTitle(rs.getString("Project Title"))
						.setCall_identifier(rs.getString("Call ID"))
						.setStartdate(rs.getString("Start Date"))
						.setEnddate(rs.getString("End Date"))
						.setEc_sc39(rs.getBoolean("ec_sc39"))
						.setOa_mandate_for_publications(rs.getBoolean("oa_mandate_for_publications"))
						.setEc_article29_3(rs.getBoolean("ec_article29_3"))
						.setDescription(rs.getString("Discipline"))
						.setLegalname(rs.getString("Organization"))
						.setCountryclass(rs.getString("Country"))
						.setRole(rs.getString("Role"))
						.setFirstname(rs.getString("Person Name"))
						.setSecondnames(rs.getString("Person Second Names"))
						.setEmail(rs.getString("Person Email"));

				writeCSVLine(out, p.asList());
			}
			out.closeEntry();
			if (log.isDebugEnabled()) {
				log.debug("Thread " + Thread.currentThread().getId() + " ends, took: " + HumanTime.exactly(System.currentTimeMillis() - start));
			}
		} finally {
			out.close();
		}
	}

	private void writeCSVLine(final ZipOutputStream out, final List<String> list) throws IOException {
		out.write(Joiner.on('\t').useForNull("").join(list).getBytes(UTF8));
		out.write('\n');
	}


	public void streamProjectDetails(final String sql, final OutputStream out, final String format) throws SQLException, IOException {
		if (log.isDebugEnabled()) {
			log.debug("Thread " + Thread.currentThread().getId() + " begin");
		}
		final long start = System.currentTimeMillis();
		int i = 0;
		try(final Connection con = getConn(); final PreparedStatement stm = getStm(sql, con); final ResultSet rs = stm.executeQuery()) {
			while (rs.next()) {
				final ProjectDetail p = getProjectDetail(rs);

				switch (format) {
				case "csv":
					out.write(p.asCSV().getBytes(UTF8));
					break;
				case "json":
					out.write(p.asJson().getBytes(UTF8));
					break;
				}
				if (++i % gzipFlushSize == 0) {
					log.debug("flushing output stream");
					out.flush();
				}
			}
			if (log.isDebugEnabled()) {
				log.debug("Thread " + Thread.currentThread().getId() + " ends, took: " + HumanTime.exactly(System.currentTimeMillis() - start));
			}
		} finally {
			if (out instanceof GZIPOutputStream) {
				((GZIPOutputStream) out).finish();
			}
			out.close();
		}
	}

	private Connection getConn() throws SQLException {
		final Connection conn = apiDataSource.getConnection();
		conn.setAutoCommit(false);
		return conn;
	}

	private PreparedStatement getStm(final String sql, final Connection con) throws SQLException {
		final PreparedStatement stm = con.prepareStatement(sql);
		stm.setFetchSize(maxRows);
		return stm;
	}

	private ProjectDetail getProjectDetail(final ResultSet rs) throws SQLException {
		return new ProjectDetail()
				.setProjectId(rs.getString("projectid"))
				.setAcronym(rs.getString("acronym"))
				.setCode(rs.getString("code"))
				.setOptional1(rs.getString("optional1"))
				.setOptional2(rs.getString("optional2"))
				.setJsonextrainfo(rs.getString("jsonextrainfo"))
				.setFundingPath(asList(rs.getArray("fundingpath")));
	}

	private List<String> asList(final Array value) throws SQLException {
		if (value != null) {
			final List<Object> list = Arrays.asList((Object[]) value.getArray());
			return Lists.newArrayList(Iterables.transform(list, new Function<Object, String>() {
				@Override
				public String apply(final Object o) {
					return o != null ? o.toString() : null;

				}
			}));
		}
		return Lists.newArrayList();
	}

}
