package eu.dnetlib.data.mdstore.modular.mongodb;

import java.util.List;
import java.util.Map;

import com.google.common.collect.Lists;
import com.mongodb.BasicDBObject;
import com.mongodb.DBObject;
import com.mongodb.WriteConcern;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.*;
import eu.dnetlib.data.mdstore.modular.RecordParser;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class MongoBulkWritesManager {

	private static final Log log = LogFactory.getLog(MongoBulkWritesManager.class);
	private final boolean discardRecords;
	private RecordParser recordParser;
	private MongoCollection<DBObject> validCollection;
	private List<WriteModel<DBObject>> validBulkOperationList;
	private int validOpCounter;
	private BulkWriteOptions writeOptions;
	private MongoCollection<DBObject> discardedCollection;
	private List<WriteModel<DBObject>> discardedBulkOperationList;
	private int discardedOpCounter;
	private int bulkSize;

	public MongoBulkWritesManager(final MongoCollection<DBObject> collection,
			final MongoCollection<DBObject> discardedCollection,
			final int bulkSize,
			final RecordParser parser,
			final boolean discardRecords) {
		this.validCollection = collection.withWriteConcern(WriteConcern.ACKNOWLEDGED);
		this.validBulkOperationList = Lists.newArrayList();
		this.validOpCounter = 0;
		this.discardedCollection = discardedCollection.withWriteConcern(WriteConcern.ACKNOWLEDGED);
		this.discardedBulkOperationList = Lists.newArrayList();
		this.discardedOpCounter = 0;
		this.bulkSize = bulkSize;
		this.recordParser = parser;
		this.discardRecords = discardRecords;
		this.writeOptions = new BulkWriteOptions().ordered(false);
	}

	public void insert(final String record) {
		try {
			final Map<String, String> recordProperties = recordParser.parseRecord(record);
			log.debug("found props: " + recordProperties);
			if (recordProperties.containsKey("id")) {
				final DBObject obj = buildDBObject(record, recordProperties);
				if (log.isDebugEnabled()) {
					log.debug("Saving object" + obj);
				}
				validBulkOperationList.add(new ReplaceOneModel(new BasicDBObject("id", obj.get("id")), obj, new UpdateOptions().upsert(true)));
				validOpCounter++;
				if (((validOpCounter % bulkSize) == 0) && (validOpCounter != 0)) {
					validCollection.bulkWrite(validBulkOperationList, writeOptions);
					validBulkOperationList = Lists.newArrayList();
				}
			} else {
				if (discardRecords) {
					log.debug("parsed record seems invalid");
					discardRecord(record);
				}
			}
		} catch (Throwable e) {
			if (discardRecords) {
				log.debug("unhandled exception: " + e.getMessage());
				discardRecord(record);
			}
		}
	}

	private void discardRecord(final String record) {
		discardedBulkOperationList.add(new InsertOneModel(new BasicDBObject("body", record)));
		discardedOpCounter++;
		if (((discardedOpCounter % bulkSize) == 0) && (discardedOpCounter != 0)) {
			discardedCollection.bulkWrite(discardedBulkOperationList, writeOptions);
			discardedBulkOperationList = Lists.newArrayList();
		}
	}

	public void flushBulks(MongoDatabase mongoDatabase) {
		//setting to journaled write concern to be sure that when the write returns everything has been flushed to disk (https://docs.mongodb.org/manual/faq/developers/#when-does-mongodb-write-updates-to-disk)
		//the explicit fsync command can't be run anymore: 'Command failed with error 13: 'fsync may only be run against the admin database.'
		if (validOpCounter != 0) {
			validCollection = getCollectionWithWriteConcern(validCollection, WriteConcern.JOURNALED);
			validCollection.bulkWrite(validBulkOperationList, writeOptions);
		}
		if (discardedOpCounter != 0) {
			discardedCollection = getCollectionWithWriteConcern(discardedCollection, WriteConcern.JOURNALED);
			discardedCollection.bulkWrite(discardedBulkOperationList, writeOptions);
		}
		//setting write concern back to ACKNOWLEDGE to avoid the execution of future writes all in Journaled mode
		validCollection = getCollectionWithWriteConcern(validCollection, WriteConcern.ACKNOWLEDGED);
		discardedCollection = getCollectionWithWriteConcern(discardedCollection, WriteConcern.ACKNOWLEDGED);
	}

	private DBObject buildDBObject(final String record, final Map<String, String> recordProperties) {
		final DBObject obj = new BasicDBObject();
		obj.put("id", recordProperties.get("id"));
		obj.put("originalId", recordProperties.get("originalId"));
		obj.put("body", record);
		obj.put("timestamp", System.currentTimeMillis());
		return obj;
	}

	private MongoCollection<DBObject> getCollectionWithWriteConcern(MongoCollection<DBObject> collection, WriteConcern writeConcern) {
		return collection.withWriteConcern(writeConcern);
	}

}
