package eu.dnetlib.data.index.solr.inspector;

import static eu.dnetlib.miscutils.collections.MappedCollection.listMap;

import java.io.IOException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.Resource;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.ws.wsaddressing.W3CEndpointReference;

import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.SolrServerException;
import org.dom4j.DocumentException;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.xml.sax.SAXException;
import org.z3950.zing.cql.CQLParseException;

import com.google.common.collect.Lists;

import eu.dnetlib.data.provision.index.rmi.IndexServiceException;
import eu.dnetlib.enabling.inspector.AbstractInspectorController;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.resultset.rmi.ResultSetException;
import eu.dnetlib.enabling.resultset.rmi.ResultSetService;
import eu.dnetlib.enabling.tools.ServiceLocator;
import eu.dnetlib.enabling.tools.ServiceResolver;
import eu.dnetlib.functionality.index.solr.SolrIndex;
import eu.dnetlib.functionality.index.solr.SolrIndexServiceImpl;
import eu.dnetlib.functionality.index.solr.query.IndexQueryFactory;
import eu.dnetlib.functionality.index.solr.query.QueryLanguage;
import eu.dnetlib.functionality.index.solr.resultset.factory.IndexResultSetFactory;
import eu.dnetlib.functionality.index.solr.suggest.Hint;
import eu.dnetlib.functionality.index.solr.utils.MetadataReference;
import eu.dnetlib.functionality.index.solr.utils.MetadataReferenceFactory;
import eu.dnetlib.miscutils.collections.Pair;
import eu.dnetlib.miscutils.datetime.HumanTime;
import eu.dnetlib.miscutils.functional.UnaryFunction;
import eu.dnetlib.miscutils.functional.string.EscapeHtml;
import eu.dnetlib.miscutils.functional.xml.IndentXmlString;

@Controller
public class SolrInspector extends AbstractInspectorController {

	private static final Log log = LogFactory.getLog(SolrInspector.class); // NOPMD by marko on 11/24/08 5:02 PM

	@Resource
	private SolrIndexServiceImpl solrIndexService;

	@Resource
	private IndexResultSetFactory indexResultSetFactory;

	@Resource
	private MetadataReferenceFactory mdFactory;

	@Resource
	private IndexQueryFactory indexQueryFactory;

	/**
	 * service resolver.
	 */
	@Resource
	private ServiceResolver serviceResolver;

	/**
	 * lookup locator.
	 */
	@Resource(name = "lookupLocator")
	private ServiceLocator<ISLookUpService> lookupLocator;

	/**
	 * result page size.
	 */
	private static final int PAGE_SIZE = 10;

	public static class SelectOption {

		private String value;
		private boolean selected;

		public SelectOption(final String value, final boolean selected) {
			super();
			this.value = value;
			this.selected = selected;
		}

		public String getValue() {
			return value;
		}

		public void setValue(final String value) {
			this.value = value;
		}

		public boolean isSelected() {
			return selected;
		}

		public void setSelected(final boolean selected) {
			this.selected = selected;
		}

	}

	public class IndexInspectorEntry {

		public String index;
		public String indexCB;

		public IndexInspectorEntry(final String index, final String indexCB) {
			this.index = index;
			this.indexCB = indexCB;
		}
	}

	@RequestMapping(value = "/inspector/solrDeleteByQuery.do", method = RequestMethod.GET)
	public void deleteByQuery(final Model model) {
		model.addAttribute("indices", getIndexList());
	}

	@RequestMapping(value = "/inspector/solrDeleteByQuery.do", method = RequestMethod.POST)
	public void processDeleteByQuery(final Model model,
			@RequestParam("query") final String query,
			@RequestParam("dsId") final String dsId,
			@RequestParam(value = "optimize", required = false) final Boolean optimizeParam,
			@RequestParam(value = "applyToAll", required = false) final Boolean applyToAllParam) throws IndexServiceException {

		boolean optimize = false;
		if ((optimizeParam != null) && (optimizeParam == true)) {
			optimize = true;
		}

		boolean applyToAll = false;
		if ((applyToAllParam != null) && (applyToAllParam == true)) {
			applyToAll = true;
		}

		// TODO handle this call through the actor
		solrIndexService.getSolrIndexServer().deleteByQuery(query, dsId, applyToAll);
		solrIndexService.getSolrIndexServer().commit(dsId);

		if (optimize) {
			solrIndexService.getSolrIndexServer().optimize(dsId);
		}

		log.info("deleted by '" + query + "' on " + dsId);

		model.addAttribute("indices", getIndexList());
		model.addAttribute("dsId", dsId);
		model.addAttribute("query", query);
		model.addAttribute("message", "Deleted by '" + query + "' on " + dsId);
	}

	@RequestMapping(value = "/inspector/solrUpdateDocuments.do")
	public void processUpdateDocuments(final Model model,
			@RequestParam(value = "query", required = false) String query,
			@RequestParam(value = "format", required = false) final String format,
			@RequestParam(value = "layout", required = false) final String layout,
			@RequestParam(value = "interp", required = false) final String interp,
			@RequestParam(value = "fields", required = false) final String fieldList,
			@RequestParam(value = "regex", required = false) final String regex,
			@RequestParam(value = "replace", required = false) final String replace) throws CQLParseException, IOException, SolrServerException,
			ParserConfigurationException, SAXException, ISLookUpException {

		if (query != null) {

			final MetadataReference mdRef = mdFactory.getMetadata(format, layout, interp);
			final SolrIndex index = solrIndexService.getSolrIndexServer().getIndexMap().getIndexByMetadata(mdRef);

			try {
				int nUpdates = solrIndexService.updateDocuments(query, mdRef, fieldList, regex, replace);
				index.getServer().commit();

				model.addAttribute("nUpdates", nUpdates);

			} catch (Exception e) {
				log.error("error updating documents on index: " + mdRef.toString(), e);
				index.getServer().rollback();
			}
		}

		if ((query == null) || query.equals("")) {
			query = "*=*";
		}

		model.addAttribute("query", query);
		model.addAttribute("formats", selectOptions(getFormats(), format));
		model.addAttribute("layouts", selectOptions(getLayouts(), layout));
		model.addAttribute("interps", selectOptions(getInterpretations(), interp));
	}

	/**
	 * search the whole index service.
	 * 
	 * @param model
	 *            mvc model
	 * @param query
	 *            query
	 * @param format
	 *            metadata format
	 * @param lang
	 *            query language
	 * @param indent
	 *            indent results flag
	 * @throws IndexServiceException
	 *             could happen
	 * @throws ResultSetException
	 *             could happen
	 * @throws ISLookUpException
	 *             could happen
	 * @throws IOException
	 * @throws CQLParseException
	 * @throws DocumentException
	 * @throws SolrServerException
	 */
	@RequestMapping(value = "/inspector/searchSolrIndex.do")
	public void search(final Model model,
			@RequestParam(value = "query", required = false) String query,
			@RequestParam(value = "idxId", required = false) final String idxId,
			@RequestParam(value = "format", required = false) final String format,
			@RequestParam(value = "layout", required = false) final String layout,
			@RequestParam(value = "interp", required = false) final String interp,
			@RequestParam(value = "language", required = false) final String lang,
			@RequestParam(value = "start", required = false) final Integer startParam,
			@RequestParam(value = "indent", required = false) Boolean indent,
			@RequestParam(value = "spellcheck", required = false) final Boolean spellcheck,
			@RequestParam(value = "pageSize", required = false) Integer pageSize,
			@RequestParam(value = "solrQuery", required = false) String solrQuery) throws IndexServiceException, ResultSetException, ISLookUpException,
			CQLParseException, IOException, SolrServerException, DocumentException {

		int total = 0;
		int start = 0;

		if (pageSize == null) {
			pageSize = PAGE_SIZE;
		}

		if (startParam != null) {
			start = startParam;
		}

		if (query != null) {
			final MetadataReference mdRef = mdFactory.getMetadata(format, layout, interp);

			if ((spellcheck != null) && spellcheck) {
				Pair<String, Hint> hint = solrIndexService.getSolrIndexServer().suggest(QueryLanguage.valueOf(lang), query, mdRef, idxId);
				String queryHint = hint.getKey();
				model.addAttribute("queryHint", StringEscapeUtils.escapeHtml(queryHint));
				model.addAttribute("autofollow", hint.getValue().isAutofollow());
			}

			try {
				solrQuery = indexQueryFactory.getIndexQuery(QueryLanguage.valueOf(lang), query, mdRef, idxId).toString();

				log.info("running " + lang + " query: " + query + " on format " + format + " with layout " + layout);

				long starting = System.currentTimeMillis();
				long elapsed = starting;
				final W3CEndpointReference epr = indexResultSetFactory.getLookupResultSet(QueryLanguage.valueOf(lang), query, mdRef, idxId);

				log.debug("GOT EPR: " + epr);

				final ResultSetService resultSet = serviceResolver.getService(ResultSetService.class, epr);
				final String rsId = serviceResolver.getResourceIdentifier(epr);

				total = resultSet.getNumberOfElements(rsId);
				if (total < pageSize) {
					pageSize = total;
				}

				List<String> elements = resultSet.getResult(rsId, 1 + start, start + pageSize, "waiting");
				elapsed = System.currentTimeMillis() - starting;

				if (elements == null) {
					elements = new ArrayList<String>();
				}

				if (indent == null) {
					indent = true;
				}

				if ((indent != null) && indent) {
					elements = listMap(elements, new IndentXmlString());
				}

				final List<String> escaped = listMap(elements, new EscapeHtml());

				model.addAttribute("results", escaped);
				model.addAttribute("elapsed", HumanTime.exactly(elapsed));
			} catch (Throwable e) {
				solrQuery = formatStackTrace(e);
			}

		}

		final List<String> idxIds = getIndexDsIds();

		idxIds.add(0, "all");

		if ((query == null) || query.equals("")) {
			query = "*=*";
		}

		model.addAttribute("encodedQuery", URLEncoder.encode(query, "UTF-8"));
		model.addAttribute("query", query);

		model.addAttribute("total", total);
		model.addAttribute("start", start);
		model.addAttribute("nextPage", start + pageSize);
		model.addAttribute("prevPage", Math.max(0, start - pageSize));

		model.addAttribute("idxIds", selectOptions(idxIds, idxId));
		model.addAttribute("formats", selectOptions(getFormats(), format));
		model.addAttribute("layouts", selectOptions(getLayouts(), layout));
		model.addAttribute("interps", selectOptions(getInterpretations(), interp));
		model.addAttribute("languages", selectOptions(getQueryLanguages(), lang));

		model.addAttribute("language", lang);
		model.addAttribute("indent", indent);
		model.addAttribute("spellcheck", spellcheck);
		model.addAttribute("solrQuery", solrQuery);
	}

	@RequestMapping(value = "/inspector/browseSolrIndex.do")
	public void browse(final Model model,
			@RequestParam(value = "query", required = false) String querySource,
			@RequestParam(value = "field", required = false) final String browseField,
			@RequestParam(value = "idxId", required = false) final String idxId,
			@RequestParam(value = "format", required = false) final String format,
			@RequestParam(value = "layout", required = false) final String layout,
			@RequestParam(value = "interpretation", required = false) final String interpretation,
			@RequestParam(value = "start", required = false) final Integer startParam,
			@RequestParam(value = "indent", required = false) Boolean indent,
			@RequestParam(value = "pageSize", required = false) Integer pageSize,
			@RequestParam(value = "solrQuery", required = false) String solrQuery) throws IndexServiceException, ResultSetException, ISLookUpException,
			SolrServerException, CQLParseException, IOException, DocumentException {

		int total = 0;
		int start = 0;

		if (pageSize == null) {
			pageSize = PAGE_SIZE;
		}

		if (startParam != null) {
			start = startParam;
		}

		if (browseField != null) {
			log.info("browsing field: " + browseField + " on format " + format + " with layout " + layout);

			String query = "query=" + querySource + "&groupby=" + browseField;
			final MetadataReference mdRef = mdFactory.getMetadata(format, layout, interpretation);

			try {
				solrQuery = indexQueryFactory.getIndexQuery(QueryLanguage.CQL, query, mdRef, idxId).toString();

				long starting = System.currentTimeMillis();
				long elapsed = starting;
				final W3CEndpointReference epr = indexResultSetFactory.getBrowsingResultSet(QueryLanguage.CQL, query, mdRef, idxId);

				log.debug("GOT EPR: " + epr);

				final ResultSetService resultSet = serviceResolver.getService(ResultSetService.class, epr);
				final String rsId = serviceResolver.getResourceIdentifier(epr);

				total = resultSet.getNumberOfElements(rsId);
				if (total < pageSize) {
					pageSize = total;
				}

				List<String> elements = resultSet.getResult(rsId, 1 + start, start + pageSize, "waiting");
				elapsed = System.currentTimeMillis() - starting;

				if (elements == null) {
					elements = new ArrayList<String>();
				}

				if (indent == null) {
					indent = true;
				}

				if ((indent != null) && indent) {
					elements = listMap(elements, new IndentXmlString());
				}

				final List<String> escaped = listMap(elements, new EscapeHtml());

				model.addAttribute("results", escaped);
				model.addAttribute("elapsed", HumanTime.exactly(elapsed));
			} catch (Throwable e) {
				solrQuery = formatStackTrace(e);
			}
		}

		final List<String> idxIds = getIndexDsIds();

		idxIds.add(0, "all");

		if ((querySource == null) || querySource.equals("")) {
			querySource = "*=*";
		}

		model.addAttribute("encodedQuerySource", URLEncoder.encode(querySource, "UTF-8"));
		model.addAttribute("querySource", querySource);
		model.addAttribute("total", total);
		model.addAttribute("start", start);
		model.addAttribute("nextPage", start + pageSize);
		model.addAttribute("prevPage", Math.max(0, start - pageSize));
		model.addAttribute("indent", indent);

		model.addAttribute("browseField", browseField);

		model.addAttribute("fields", selectOptions(getBrowsingFields(), browseField));
		model.addAttribute("idxIds", selectOptions(idxIds, idxId));
		model.addAttribute("formats", selectOptions(getFormats(), format));
		model.addAttribute("layouts", selectOptions(getLayouts(), layout));
		model.addAttribute("solrQuery", solrQuery);
	}

	private String formatStackTrace(final Throwable e) {
		return "ERROR\n" + ExceptionUtils.getFullStackTrace(e);
	}

	/**
	 * Given an list of values, return a list of SelectOption instances which have the "selected" boolean field set to true only for the
	 * element matching "current".
	 * 
	 * @param input
	 *            list of input strings
	 * @param current
	 *            current value to select
	 * @return
	 */
	private List<SelectOption> selectOptions(final List<String> input, final String current) {
		return listMap(input, new UnaryFunction<SelectOption, String>() {

			@Override
			public SelectOption evaluate(final String value) {
				return new SelectOption(value, value.equals(current));
			}
		});
	}

	private List<String> getQueryLanguages() {
		return listMap(QueryLanguage.values(), new UnaryFunction<String, QueryLanguage>() {

			@Override
			public String evaluate(final QueryLanguage lang) {
				return lang.name();
			}
		});
	}

	private List<String> getInterpretations() throws ISLookUpException {
		return lookupLocator.getService().quickSearchProfile(
				"distinct-values(//RESOURCE_PROFILE[.//RESOURCE_TYPE/@value = 'IndexDSResourceType']//METADATA_FORMAT_INTERPRETATION/text())");
	}

	private List<String> getFormats() throws ISLookUpException {
		return lookupLocator.getService().quickSearchProfile(
				"distinct-values(//RESOURCE_PROFILE[.//RESOURCE_TYPE/@value = 'IndexDSResourceType']//METADATA_FORMAT/text())");
	}

	private List<String> getLayouts() throws ISLookUpException {
		return lookupLocator.getService().quickSearchProfile("distinct-values(for $x in //RESOURCE_PROFILE//LAYOUT/@name/string() order by $x return $x)");
	}

	private List<String> getIndexDsIds() throws ISLookUpException {
		return lookupLocator.getService().quickSearchProfile("//RESOURCE_IDENTIFIER[../RESOURCE_TYPE/@value = 'IndexDSResourceType']/@value/string()");
	}

	private List<String> getBrowsingFields() throws ISLookUpException {
		return lookupLocator.getService().quickSearchProfile("distinct-values(//RESOURCE_PROFILE//LAYOUT//FIELD/@browsingAliasFor/string())");
	}

	private List<IndexInspectorEntry> getIndexList() {
		List<IndexInspectorEntry> indicesCBox = Lists.newArrayList();

		for (String id : solrIndexService.getListOfIndices()) {
			String cb = "[" + solrIndexService.getSolrIndexServer().getIndexMap().getMdRefById(id).toString() + "] - " + id;
			indicesCBox.add(new IndexInspectorEntry(id, cb));
		}

		return indicesCBox;
	}

}
