package eu.dnetlib.enabling.database.resultset;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.jdbc.support.rowset.SqlRowSet;

import eu.dnetlib.enabling.database.rmi.DatabaseException;
import eu.dnetlib.enabling.database.utils.DatabaseUtils;
import eu.dnetlib.enabling.resultset.ResultSetListener;

public class SQLResultSetListener implements ResultSetListener {

	private String db;
	private String sql;
	private DatabaseUtils dbUtils;

	private int lastToPosition = 0;
	private SqlRowSet rowSet = null;

	private Integer size = null;

	private static final Log log = LogFactory.getLog(SQLResultSetListener.class); // NOPMD by marko on 11/24/08 5:02 PM

	protected SQLResultSetListener() {
	}

	protected SQLResultSetListener(String db, String sql, DatabaseUtils dbUtils) {
		super();
		this.db = db;
		this.sql = sql;
		this.dbUtils = dbUtils;
	}

	@Override
	public List<String> getResult(int fromPosition, int toPosition) {
		if (fromPosition < 1) {
			throw new IllegalArgumentException("fromPosition must be >= 1");
		} else if (toPosition < fromPosition) {
			throw new IllegalArgumentException("toPosition must be equal or greater than fromPosition");
		}
		int limit = toPosition - fromPosition + 1;

		try {
			List<String> list = new ArrayList<String>();

			final SqlRowSet rows = getPage(fromPosition, toPosition);
			while (rows.next() && limit-- > 0) {
				Map<?, ?> o = parseRowSet(rows);
				if (o instanceof Map<?, ?>) {
					list.add(dbUtils.rowToDocument((Map<?, ?>) o).asXML());
				}
			}

			lastToPosition = toPosition;

			return list;
		} catch (DatabaseException e) {
			log.error("Error in getResult " + fromPosition + "-" + toPosition, e);
			throw new IllegalStateException("Error in getResult " + fromPosition + "-" + toPosition, e);
		}
	}

	private SqlRowSet getPage(int fromPosition, int toPosition) throws DatabaseException {
		if (fromPosition == lastToPosition + 1 && rowSet != null) {
			log.debug("returning old rowset");
			rowSet.previous();
			return rowSet;
		}

		final int offset = fromPosition - 1;

		String query = sql;
		if (offset > 0) {
			query += " OFFSET " + offset;
		}
		this.rowSet = dbUtils.executeSql(db, query, SqlRowSet.class);
		return rowSet;
	}

	private Map<?, ?> parseRowSet(SqlRowSet rows) {
		HashMap<String, Object> res = new HashMap<String, Object>();
		for (String column : rows.getMetaData().getColumnNames())
			res.put(column, rows.getObject(column));
		return res;
	}

	@Override
	public int getSize() {
		if (size != null) {
			return size;
		}

		String query = "SELECT count(*) FROM ( " + sql + " ) AS TABLELISTENER";

		try {
			this.size = dbUtils.executeSql(db, query, Integer.class);
			return size;
		} catch (DatabaseException e) {
			log.error("Error in getSize, query: " + query, e);
			throw new IllegalStateException("Error in getSize, query: " + query, e);
		}
	}

	public String getDb() {
		return db;
	}

	public void setDb(String db) {
		this.db = db;
	}

	public String getSql() {
		return sql;
	}

	public void setSql(String sql) {
		this.sql = sql;
	}

	public DatabaseUtils getDbUtils() {
		return dbUtils;
	}

	public void setDbUtils(DatabaseUtils dbUtils) {
		this.dbUtils = dbUtils;
	}

}
