package eu.dnetlib.data.information.oai.publisher.core;

import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;

import eu.dnetlib.data.information.oai.publisher.CannotDisseminateFormatException;
import eu.dnetlib.data.information.oai.publisher.OaiPublisherException;
import eu.dnetlib.data.information.oai.publisher.OaiPublisherRuntimeException;
import eu.dnetlib.data.information.oai.publisher.conf.ISLookUpClient;
import eu.dnetlib.data.information.oai.publisher.conf.OAIConfigurationExistReader;
import eu.dnetlib.data.information.oai.publisher.info.ListDocumentsInfo;
import eu.dnetlib.data.information.oai.publisher.info.ListRecordsInfo;
import eu.dnetlib.data.information.oai.publisher.info.MDFInfo;
import eu.dnetlib.data.information.oai.publisher.info.RecordInfo;
import eu.dnetlib.data.information.oai.publisher.info.ResumptionToken;
import eu.dnetlib.data.information.oai.publisher.info.SetInfo;
import eu.dnetlib.data.information.oai.sets.SetCollection;
import eu.dnetlib.miscutils.cache.EhCache;

/**
 * Core OAI servlet implementation.
 * 
 * @author michele,alessia
 * 
 */

public abstract class AbstractOAICore {

	private static final Log log = LogFactory.getLog(AbstractOAICore.class); // NOPMD by marko on 11/24/08 5:02 PM

	private String currentDBName;

	@Autowired
	private SetCollection setCollection;

	@Autowired
	private OAIConfigurationExistReader oaiConfigurationExistReader;

	@Autowired
	private ISLookUpClient lookupClient;

	@Resource
	private EhCache<String, MDFInfo> mdFormatsCache;

	/**
	 * page size.
	 */
	protected int pageSize = 100;

	/**
	 * Returns informations about a record. It contains basic information about a record, such as "prefix", "identifier", "datestamp",
	 * "setspec", "metadata".
	 * 
	 * @param identifier
	 *            record identifier
	 * @param prefix
	 *            metadata prefix
	 * @return a recordInfo instance
	 * @throws OaiPublisherException
	 *             could happen
	 */
	public RecordInfo getInfoRecord(final String identifier, final String prefix) throws OaiPublisherException {
		setCurrentDBFromIS(false);
		MDFInfo mdf = obtainMDFInfo(prefix);
		return getRecordById(mdf, identifier);
	}

	/**
	 * List records.
	 * 
	 * @param onlyIdentifiers
	 *            only return record identifiers
	 * @param metadataPrefix
	 *            metadata prefix
	 * @param set
	 *            set name
	 * @param from
	 *            from date
	 * @param until
	 *            to date
	 * @return ListRecordsInfo
	 * @throws OaiPublisherException
	 *             could happen
	 */
	public ListRecordsInfo listRecords(final boolean onlyIdentifiers, final String metadataPrefix, final String set, final String from, final String until)
			throws OaiPublisherException {
		setCurrentDBFromIS(false);
		final ListRecordsInfo res = new ListRecordsInfo();

		if (from != null) {
			res.setFrom(from);
			res.setWithSize(false);
		}
		if (until != null) {
			res.setUntil(until);
			res.setWithSize(false);
		}
		if (metadataPrefix != null) {
			res.setMetadataprefix(metadataPrefix);
		}
		if (set != null) {
			res.setSet(set);
		}

		ListDocumentsInfo documents = getDocuments(onlyIdentifiers, set, metadataPrefix, from, until);
		ResumptionToken resumptionToken = documents.getResumptionToken();
		int cursor = documents.getCursor();
		int nMaxElements = documents.getnMaxElements();

		if (onlyIdentifiers) {
			res.setIdentifiers(documents.getDocs());
		} else {
			res.setDocuments(documents.getDocs());
		}

		if ((resumptionToken != null)) {
			res.setResumptiontoken(resumptionToken.serialize());
			res.setCursor(cursor);
			res.setSize(nMaxElements);
		}
		return res;
	}

	/**
	 * List records.
	 * 
	 * @param onlyIdentifiers
	 *            only resource identifiers.
	 * @param resumptionTokenParameter
	 *            resumption token
	 * @return ListRecordsInfo
	 * @throws OaiPublisherException
	 *             could happen
	 */
	public ListRecordsInfo listRecords(final boolean onlyIdentifiers, final String resumptionToken) throws OaiPublisherException {
		setCurrentDBFromIS(false);
		ListDocumentsInfo docs = getDocuments(onlyIdentifiers, resumptionToken);

		final ListRecordsInfo res = new ListRecordsInfo();

		if (docs.getMetadataPrefix() != null) {
			res.setMetadataprefix(docs.getMetadataPrefix());
		}

		if (onlyIdentifiers) {
			res.setIdentifiers(docs.getDocs());
		} else {
			res.setDocuments(docs.getDocs());
		}

		ResumptionToken newResumptionToken = docs.getResumptionToken();
		if ((newResumptionToken != null)) {
			res.setResumptiontoken(newResumptionToken.serialize());
			res.setCursor(docs.getCursor());
			res.setSize(docs.getnMaxElements());
			res.setWithSize(!newResumptionToken.hasDateRange());
		}
		return res;
	}

	public List<? extends SetInfo> listSets() throws OaiPublisherException {
		setCurrentDBFromIS(false);
		return this.setCollection.getAllSets(true, getCurrentDBName());
	}

	public List<MDFInfo> listMetadataFormats() throws OaiPublisherException {
		setCurrentDBFromIS(false);
		return this.oaiConfigurationExistReader.getMetadataFormatInfo(true);
	}

	public boolean existSet(final String setSpec) {
		try {
			setCurrentDBFromIS(false);
		} catch (OaiPublisherException e) {
			throw new OaiPublisherRuntimeException(e);
		}
		return this.setCollection.containEnabledSet(setSpec, getCurrentDBName());
	}

	protected MDFInfo obtainMDFInfo(final String metadataPrefix) throws OaiPublisherException {
		MDFInfo mdf = this.mdFormatsCache.get(metadataPrefix);
		if (mdf == null) {
			log.fatal("Not using cache for " + metadataPrefix);
			mdf = oaiConfigurationExistReader.getMetadataFormatInfo(metadataPrefix);
			if (mdf == null) throw new CannotDisseminateFormatException("Invalid metadataPrefix " + metadataPrefix);
			this.mdFormatsCache.put(metadataPrefix, mdf);
		}
		return mdf;
	}

	/**
	 * Set the current DB from the configuration on the IS.
	 * <p>
	 * If force is false, the value is updated only if blank
	 * 
	 * @param force
	 *            true to update the currentDB when the current value is not blank
	 * @throws OaiPublisherException
	 */
	protected void setCurrentDBFromIS(final boolean force) throws OaiPublisherException {
		try {
			if (force || StringUtils.isBlank(currentDBName)) {
				currentDBName = getLookupClient().getCurrentDB();
				log.fatal("Set " + currentDBName + " as current DB");
			}
		} catch (Exception e) {
			log.error("Can't set the current db for OAI, check configuration");
			throw new OaiPublisherException(e);
		}
	}

	protected abstract RecordInfo getRecordById(final MDFInfo mdf, final String id) throws OaiPublisherException;

	protected abstract ListDocumentsInfo getDocuments(final boolean onlyIdentifiers,
			final String set,
			final String metadataPrefix,
			final String from,
			final String until) throws OaiPublisherException;

	protected abstract ListDocumentsInfo getDocuments(final boolean onlyIdentifiers, final String resumptionToken) throws OaiPublisherException;

	public String getCurrentDBName() {
		return currentDBName;
	}

	public void setCurrentDBName(final String currentDBName) {
		this.currentDBName = currentDBName;
	}

	public SetCollection getSetCollection() {
		return setCollection;
	}

	public void setSetCollection(final SetCollection setCollection) {
		this.setCollection = setCollection;
	}

	public OAIConfigurationExistReader getOaiConfigurationExistReader() {
		return oaiConfigurationExistReader;
	}

	public void setOaiConfigurationExistReader(final OAIConfigurationExistReader oaiConfigurationExistReader) {
		this.oaiConfigurationExistReader = oaiConfigurationExistReader;
	}

	public ISLookUpClient getLookupClient() {
		return lookupClient;
	}

	public void setLookupClient(final ISLookUpClient lookupClient) {
		this.lookupClient = lookupClient;
	}

	public EhCache<String, MDFInfo> getMdFormatsCache() {
		return mdFormatsCache;
	}

	public void setMdFormatsCache(final EhCache<String, MDFInfo> mdFormatsCache) {
		this.mdFormatsCache = mdFormatsCache;
	}
}
