package eu.dnetlib.msro.openaireplus.workflows.nodes.index;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.List;

import javax.annotation.Resource;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.core.io.ClassPathResource;

import com.google.common.collect.Iterables;
import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.NodeToken;

import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpDocumentNotFoundException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.tools.ServiceLocator;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.miscutils.functional.hash.Hashing;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;

public class PrepareIndexDataJobNode extends SimpleJobNode {

	/**
	 * logger.
	 */
	private static final Log log = LogFactory.getLog(PrepareIndexDataJobNode.class); // NOPMD by marko on 11/24/08 5:02 PM

	@Resource(name = "lookupLocator")
	private ServiceLocator<ISLookUpService> lookupLocator;

	@Autowired
	private ContextLoader contextLoader;

	@Autowired
	private EntityGrouperConfigurationLoader grouperConfLoader;

	/**
	 * Stylesheet which transforms a layout to another stylesheet which converts a input record to a index record.
	 */
	private String layoutToRecordStylesheet;

	private String outputRecordsPathParam;

	private String rottenRecordsPathParam;

	private String hbaseTable;

	@Override
	protected String execute(NodeToken token) throws Exception {

		log.info("start preparing job");

		final String xslt = prepareXslt(env("format", token), env("layout", token));

		token.getEnv().setAttribute("contextmap", contextLoader.load());
		token.getEnv().setAttribute("index.entity.links", grouperConfLoader.load());

		token.getEnv().setAttribute("index.xslt", xslt);

		token.getEnv().setAttribute(getOutputRecordsPathParam(), "/tmp" + getFileName(token, "indexrecords"));
		token.getEnv().setAttribute(getRottenRecordsPathParam(), "/tmp" + getFileName(token, "rottenrecords"));

		token.getEnv().setAttribute("index.solr.url.zk", getIndexSolrUrlZk());
		token.getEnv().setAttribute("index.solr.url.local", getIndexSolrUrlLocal());
		token.getEnv().setAttribute("index.solr.url.list", getIndexSolrUrlList());

		token.getEnv().setAttribute("index.shutdown.wait.time", getIndexSolrShutdownWait());
		token.getEnv().setAttribute("index.buffer.flush.threshold", getIndexBufferFlushTreshold());
		token.getEnv().setAttribute("index.solr.local.feeding", isIndexLocalFeeding());
		token.getEnv().setAttribute("index.solr.sim.mode", isFeedingSimulationMode());

		token.getEnv().setAttribute("index.solr.collection", getCollectionName(token));
		token.getEnv().setAttribute("index.feed.timestamp", DateUtils.now_ISO8601());

		return Arc.DEFAULT_ARC;
	}

	public String getIndexSolrUrlZk() throws ISLookUpException {
		return getServiceConfigValue("for $x in /RESOURCE_PROFILE[.//RESOURCE_TYPE/@value='IndexServiceResourceType'] return $x//PROTOCOL[./@name='SOLR']/@address/string()");
	}

	public String getIndexSolrUrlLocal() throws ISLookUpException {
		return queryForServiceProperty("hadoopLocalUrl");
	}

	public String getIndexSolrUrlList() throws ISLookUpException {
		return queryForServiceProperty("clientPoolUrls");
	}

	public String getIndexSolrShutdownWait() throws ISLookUpException {
		return queryForServiceProperty("feedingShutdownTolerance");
	}

	public String getIndexBufferFlushTreshold() throws ISLookUpException {
		return queryForServiceProperty("feedingBufferFlushThreshold");
	}

	public String isIndexLocalFeeding() throws ISLookUpException {
		return queryForServiceProperty("feedingOnLocalUrls");
	}

	public String isFeedingSimulationMode() throws ISLookUpException {
		return queryForServiceProperty("feedingSimulationMode");
	}

	private String queryForServiceProperty(String key) throws ISLookUpException {
		return getServiceConfigValue("for $x in /RESOURCE_PROFILE[.//RESOURCE_TYPE/@value='IndexServiceResourceType'] return $x//SERVICE_PROPERTIES/PROPERTY[./@ key='"
				+ key + "']/@value/string()");
	}

	private String getServiceConfigValue(final String xquery) throws ISLookUpException {
		log.debug("quering for service property: " + xquery);
		final List<String> urls = lookupLocator.getService().quickSearchProfile(xquery);
		if (urls == null || urls.size() != 1) { throw new IllegalStateException("unable to find unique service property, xquery: " + xquery); }
		return Iterables.getOnlyElement(urls);
	}

	private String getFileName(final NodeToken token, final String fileNamePrefix) {
		return "/" + fileNamePrefix + "_" + getHbaseTable() + "_" + token.getEnv().getAttribute("format") + ".seq";
	}

	private String getCollectionName(NodeToken token) {
		return env("format", token) + "-" + env("layout", token) + "-" + env("interpretation", token);
	}

	private String env(String s, NodeToken token) {
		return token.getEnv().getAttribute(s);
	}

	/**
	 * Transforms each OAF record into a index record.
	 * 
	 * @param mdStoreRsetEpr
	 *            mdstore resulsetset
	 * @param layout
	 *            layout
	 * @param format
	 *            format
	 * @return resultset with transformed records
	 * @throws ISLookUpException
	 *             could happen
	 * @throws IOException
	 *             could happen
	 * @throws TransformerException
	 *             could happen
	 */
	protected String prepareXslt(final String format, final String layout) throws ISLookUpException, IOException, TransformerException {

		final TransformerFactory factory = TransformerFactory.newInstance();
		final Transformer layoutTransformer = factory.newTransformer(new StreamSource(new StringReader(readXslt(getLayoutToRecordStylesheet()))));

		final StreamResult layoutToXsltXslt = new StreamResult(new StringWriter());

		layoutTransformer.setParameter("format", format);
		layoutTransformer.transform(new StreamSource(new StringReader(getLayoutSource(format, layout))), layoutToXsltXslt);

		return new String(Hashing.encodeBase64(layoutToXsltXslt.getWriter().toString()));
	}

	private String readXslt(String s) throws IOException {
		ClassPathResource resource = new ClassPathResource(s);
		InputStream inputStream = resource.getInputStream();
		return IOUtils.toString(inputStream);
	}

	private String getLayoutSource(final String format, final String layout) throws ISLookUpDocumentNotFoundException, ISLookUpException {
		return lookupLocator.getService().getResourceProfileByQuery(
				"collection('')//RESOURCE_PROFILE[.//RESOURCE_TYPE/@value = 'MDFormatDSResourceType' and .//NAME='" + format + "']//LAYOUT[@name='" + layout
						+ "']");
	}

	public String getLayoutToRecordStylesheet() {
		return layoutToRecordStylesheet;
	}

	public void setLayoutToRecordStylesheet(final String layoutToRecordStylesheet) {
		this.layoutToRecordStylesheet = layoutToRecordStylesheet;
	}

	public String getHbaseTable() {
		return hbaseTable;
	}

	@Required
	public void setHbaseTable(String hbaseTable) {
		this.hbaseTable = hbaseTable;
	}

	public String getOutputRecordsPathParam() {
		return outputRecordsPathParam;
	}

	public void setOutputRecordsPathParam(String outputRecordsPathParam) {
		this.outputRecordsPathParam = outputRecordsPathParam;
	}

	public String getRottenRecordsPathParam() {
		return rottenRecordsPathParam;
	}

	public void setRottenRecordsPathParam(String rottenRecordsPathParam) {
		this.rottenRecordsPathParam = rottenRecordsPathParam;
	}

}
