package eu.dnetlib.data.collector.plugins.ftp;

import java.io.IOException;
import java.util.Collection;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.net.ftp.FTPFile;
import org.apache.commons.net.ftp.FTPFileFilter;

/**
 * FTPDirectoryWalker runs recursively under a FTP directory structure starting from a given path.
 * 
 * Acts as a producer.
 * 
 * @author Sandro
 * 
 * @param <T>
 *            Type of expected content to extract from files NB. Generally strings.
 */

public abstract class FtpDirectoryWalker<T> {

	/**
	 * logger
	 */
	private static final Log log = LogFactory.getLog(FtpDirectoryWalker.class); // NOPMD by marko on 11/24/08 5:02 PM

	/**
	 * The file filter to use to filter files and directories.
	 */
	// private final FTPFileFilter filter;

	/**
	 * The limit on the directory depth to walk.
	 */
	private final int depthLimit;

	private ItemUtility itemParam;

	protected FtpClientProvider clientProvider;

	private FTPFileFilter filter;

	/**
	 * Construct an FTP instance with no filtering and unlimited <i>depth</i>.
	 * 
	 * @param server
	 *            the server fTP to connect
	 * @param username
	 *            the username parameter for the fTP connection, if null the connection is anonymous
	 * @param password
	 *            the password parameter for the FTP connection
	 */
	protected FtpDirectoryWalker(final ItemUtility itemParam) {
		this(null, 0, itemParam, "ftp");
	}

	/**
	 * Construct an instance with a filter and limit the <i>depth</i> navigated to.
	 * <p>
	 * The filter controls which files and directories will be navigated to as part of the walk.
	 * 
	 * @param filter
	 *            the filter to apply, null means visit all files
	 * @param depthLimit
	 *            controls how <i>deep</i> the hierarchy is navigated to (less than 0 means unlimited)
	 * @param server
	 *            the server fTP to connect
	 * @param username
	 *            the username parameter for the fTP connection, if null the connection is anonymous
	 * @param password
	 *            the password parameter for the FTP connection
	 * 
	 */
	protected FtpDirectoryWalker(final FTPFileFilter filter, final int depthLimit, final ItemUtility itemParam, final String protocol) {
		this.filter = filter;
		this.depthLimit = depthLimit;
		this.itemParam = itemParam;

		clientProvider = (FtpClientProvider) ClientProviderFactory.getProvider("ftp");

	}

	/**
	 * Overridable callback method invoked at the start of processing.
	 * <p>
	 * This implementation does nothing.
	 * 
	 * @param startDirectory
	 *            the directory to start from
	 * @param results
	 *            the collection of result objects, may be updated
	 * @throws IOException
	 *             if an I/O Error occurs
	 */
	protected void handleStart(final String startDirectory, final Collection<T> results) throws IOException {
		// do nothing - overridable by subclass
	}

	/**
	 * Overridable callback method invoked to determine if a directory should be processed.
	 * <p>
	 * This method returns a boolean to indicate if the directory should be examined or not. If you return false, the entire directory and
	 * any subdirectories will be skipped. Note that this functionality is in addition to the filtering by file filter.
	 * <p>
	 * This implementation does nothing and returns true.
	 * 
	 * @param directory
	 *            the current directory being processed
	 * @param depth
	 *            the current directory level (starting directory = 0)
	 * @param results
	 *            the collection of result objects, may be updated
	 * @return true to process this directory, false to skip this directory
	 * @throws IOException
	 *             if an I/O Error occurs
	 */
	protected boolean handleDirectory(final String directory, final int depth, final Collection<T> results) throws IOException {
		// do nothing - overridable by subclass
		return true; // process directory
	}

	/**
	 * Overridable callback method invoked at the start of processing each directory.
	 * <p>
	 * This implementation does nothing.
	 * 
	 * @param directory
	 *            the current directory being processed
	 * @param depth
	 *            the current directory level (starting directory = 0)
	 * @param results
	 *            the collection of result objects, may be updated
	 * @throws IOException
	 *             if an I/O Error occurs
	 */
	protected void handleDirectoryStart(final String directory, final int depth, final Collection<T> results) throws IOException {
		// do nothing - overridable by subclass
	}

	/**
	 * Overridable callback method invoked for each (non-directory) file.
	 * <p>
	 * This implementation does nothing.
	 * 
	 * @param file
	 *            the current file being processed
	 * @param depth
	 *            the current directory level (starting directory = 0)
	 * @param results
	 *            the collection of result objects, may be updated
	 * @throws IOException
	 *             if an I/O Error occurs
	 */
	protected void handleFile(final FTPFile file, final int depth, final Collection<T> results) throws IOException {
		// do nothing - overridable by subclass
	}

	/**
	 * Overridable callback method invoked at the end of processing.
	 * <p>
	 * This implementation does nothing.
	 * 
	 * @param results
	 *            the collection of result objects, may be updated
	 * @throws IOException
	 *             if an I/O Error occurs
	 */
	protected void handleEnd(final Collection<T> results) throws IOException {
		// do nothing - overridable by subclass
		clientProvider.disconnect();
	}

	public void initialize() {
		clientProvider.setItemParam(itemParam);
		clientProvider.connect();
		clientProvider.changeWorkingDirectory(itemParam.getBasePath());

	}

	/**
	 * Internal method that walks the directory hierarchy in a depth-first manner.
	 * <p>
	 * Users of this class do not need to call this method. This method will be called automatically by another (public) method on the
	 * specific subclass.
	 * <p>
	 * Writers of subclasses should call this method to start the directory walk. Once called, this method will emit events as it walks the
	 * hierarchy. The event methods have the prefix <code>handle</code>.
	 * 
	 * @param startDirectory
	 *            the directory to start from, not null
	 * @param results
	 *            the collection of result objects, may be updated
	 * @throws NullPointerException
	 *             if the start directory is null
	 * @throws IOException
	 *             if an I/O Error occurs
	 */
	protected final void walk(final String startDirectory, final Collection<T> results) throws IOException {
		if (startDirectory == null) { throw new NullPointerException("Start Directory is null"); }
		handleStart(startDirectory, results);

		walk(startDirectory, -1, results);
		handleEnd(results);

	}

	/**
	 * Main recursive method to examine the directory hierarchy.
	 * 
	 * @param directory
	 *            the directory to examine, not null
	 * @param depth
	 *            the directory level (starting directory = 0)
	 * @param results
	 *            the collection of result objects, may be updated
	 * @throws IOException
	 *             if an I/O Error occurs
	 */
	private void walk(final String directory, final int depth, final Collection<T> results) throws IOException {
		if (handleDirectory(directory, depth, results)) {
			String currentDirectory = clientProvider.printWorkingDirectory();
			clientProvider.changeWorkingDirectory(directory);
			int childDepth = depth + 1;
			if ((depthLimit < 0) || (childDepth <= depthLimit)) {
				// TODO: must be implemented the filter of file
				FTPFile[] childFiles = clientProvider.listFiles();
				if (childFiles != null) {
					for (FTPFile childFile : childFiles) {
						if (childFile.isDirectory()) {
							handleDirectoryStart(clientProvider.printWorkingDirectory() + "/" + childFile.getName(), childDepth, results);
							walk(clientProvider.printWorkingDirectory() + "/" + childFile.getName(), childDepth, results);
							clientProvider.changeWorkingDirectory(currentDirectory);
						} else {
							if ((filter == null) || filter.accept(childFile)) {
								log.info("Take the File " + childFile.getName());
								handleFile(childFile, childDepth, results);
							}

						}
					}
				}
			}
			handleDirectoryEnd(directory, depth, results);
		}

	}

	protected String getCurrentDirectory() throws IOException {
		return clientProvider.printWorkingDirectory();

	}

	/**
	 * Overridable callback method invoked at the end of processing each directory.
	 * <p>
	 * This implementation does nothing.
	 * 
	 * @param directory
	 *            the directory being processed
	 * @param depth
	 *            the current directory level (starting directory = 0)
	 * @param results
	 *            the collection of result objects, may be updated
	 * @throws IOException
	 *             if an I/O Error occurs
	 */
	protected void handleDirectoryEnd(final String directory, final int depth, final Collection<T> results) throws IOException {
		// do nothing - overridable by subclass
	}

	protected String getFileFrom(final String remote) throws IOException {
		return clientProvider.retrieveFileStream(remote);
	}

	protected void completePendingCommand() throws IOException {
		clientProvider.completePendingCommand();
	}

}
