package eu.dnetlib.data.collector.plugins.filesystem;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.concurrent.BlockingQueue;

import org.apache.commons.io.DirectoryWalker;
import org.apache.commons.io.filefilter.FileFilterUtils;
import org.apache.commons.io.filefilter.HiddenFileFilter;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.apache.commons.io.filefilter.SuffixFileFilter;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * FileWalker runs recursively under a directory structure starting from a given path and for each file reads it's content and puts it in a
 * shared queue.
 * 
 * Acts as a producer.
 * 
 * @author claudio
 * 
 * @param <T>
 *            Type of expected content to extract from files NB. Generally strings.
 */
public abstract class FileWalker<T> extends DirectoryWalker<T> {

	/**
	 * Logger.
	 */
	private static final Log log = LogFactory.getLog(FileWalker.class);

	/**
	 * Shared queue.
	 */
	private final BlockingQueue<T> queue;

	/**
	 * Reference to the starting directory
	 */
	private final File source;

	/**
	 * specifies the file type;
	 */
	private final FileType type;

	public enum FileType {
		TEXT, PDF, DOC
	}

	public static String IGNORE_PREFIX = ".";

	public static String IGNORE_SUFFIX = "~";

	static IOFileFilter fileFilter = FileFilterUtils.notFileFilter(FileFilterUtils.or(new SuffixFileFilter("~"), HiddenFileFilter.HIDDEN));

	/**
	 * Builds a FileWalker.
	 * 
	 * @param sourcePath
	 *            the
	 * @param queue
	 */
	public FileWalker(final BlockingQueue<T> queue, final FileType type, final String basePath) {
		super(fileFilter, -1);
		this.source = new File(basePath);
		this.type = type;
		this.queue = queue;
	}

	@Override
	protected void handleFile(final File file, final int depth, final Collection<T> results) throws IOException {
		enqueue((BlockingQueue<T>) results, readFile(file));
	}

	@Override
	protected boolean handleDirectory(final File directory, final int depth, final Collection<T> results) throws IOException {
		if (directory.getName().startsWith(IGNORE_PREFIX)) return false;
		return super.handleDirectory(directory, depth, results);
	}

	/* Helpers */
	/**
	 * given a file, return its content as a T
	 * 
	 * @param file
	 *            the source
	 * @return the file content as an object of type T
	 * @throws IOException
	 */
	protected abstract T readFile(final File file) throws IOException;

	/**
	 * Wrapper method, starts the walk and when it's done adds the flag to the queue.
	 */
	protected abstract void doWalk() throws IOException;

	/**
	 * Adds the element to the queue
	 */
	protected void enqueue(final BlockingQueue<T> queue, final T element) {
		try {
			queue.put(element);
		} catch (InterruptedException e) {
			log.warn("ops... ", e);
		}
	}

	public BlockingQueue<T> getQueue() {
		return queue;
	}

	public File getSource() {
		return source;
	}

	public FileType getType() {
		return type;
	}

}
