package eu.dnetlib.enabling.datasources;

import java.io.StringReader;
import java.io.StringWriter;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Required;

import com.google.common.collect.Lists;

import eu.dnetlib.enabling.datasources.rmi.DatasourceDesc;
import eu.dnetlib.enabling.datasources.rmi.DatasourceManagerService;
import eu.dnetlib.enabling.datasources.rmi.DatasourceManagerServiceException;
import eu.dnetlib.enabling.datasources.rmi.IfaceDesc;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.is.registry.rmi.ISRegistryService;
import eu.dnetlib.enabling.tools.AbstractBaseService;
import eu.dnetlib.enabling.tools.ServiceLocator;

public class DatasourceManagerServiceImpl extends AbstractBaseService implements DatasourceManagerService {

	private ServiceLocator<ISRegistryService> registryLocator;
	private ServiceLocator<ISLookUpService> lookupLocator;

	private static final Log log = LogFactory.getLog(DatasourceManagerServiceImpl.class);
	private static final String REPOSITORY_SERVICE_RESOURCE_TYPE = "RepositoryServiceResourceType";

	private static SAXReader reader = new SAXReader();

	@Override
	public boolean addDatasource(final DatasourceDesc ds) throws DatasourceManagerServiceException {
		try {

			final String profile = SimpleDatasourceMagerUtils.toDatasourceProfile(ds);
			log.debug("Generated profile: " + profile);
			final String profId = registryLocator.getService().registerProfile(profile);
			log.info("A new datasource has been registered with profile ID: " + profId);
			return true;
		} catch (Exception e) {
			log.error("Error adding a datasource", e);
			throw new DatasourceManagerServiceException("Error adding a datasource", e);
		}
	}

	@Override
	public boolean deleteDatasource(final String dsId) throws DatasourceManagerServiceException {
		try {
			return getRegistryLocator().getService().deleteProfile(findProfileId(dsId));
		} catch (Exception e) {
			log.error("Error deleting profile: " + dsId);
			throw new DatasourceManagerServiceException("Error deleting profile: " + dsId, e);
		}

	}

	@Override
	public DatasourceDesc getDatasource(final String dsId) throws DatasourceManagerServiceException {
		try {
			return SimpleDatasourceMagerUtils.toDatasourceDesc(findProfile(dsId));
		} catch (Exception e) {
			log.error("Error fetching datasource " + dsId);
			throw new DatasourceManagerServiceException("Error fetching datasource " + dsId, e);
		}
	}

	@Override
	public List<DatasourceDesc> listAllDatasources() throws DatasourceManagerServiceException {
		final String xquery = "/*[.//RESOURCE_TYPE/@value='RepositoryServiceResourceType']";
		return listDatasourcesUsingQuery(xquery);
	}

	@Override
	public List<DatasourceDesc> listDatasourcesUsingFilter(final String compliance,
			final String contentDescription,
			final String iisProcessingWorkflow,
			final String collectedFrom) throws DatasourceManagerServiceException {

		final StringWriter swCond = new StringWriter();
		swCond.append(".//RESOURCE_TYPE/@value='RepositoryServiceResourceType'");

		if ((compliance != null) && !compliance.isEmpty()) {
			swCond.append(" and .//INTERFACE/@compliance='" + compliance + "'");
		}
		if ((contentDescription != null) && !contentDescription.isEmpty()) {
			swCond.append(" and .//INTERFACE/@contentDescription='" + contentDescription + "'");
		}
		if ((iisProcessingWorkflow != null) && !iisProcessingWorkflow.isEmpty()) {
			swCond.append(" and .//FIELD[./key='iis_processing_workflow']/value='" + iisProcessingWorkflow + "'");
		}
		if ((collectedFrom != null) && !collectedFrom.isEmpty()) {
			swCond.append(" and .//DATASOURCE_ORIGINAL_ID/@provenance = '" + collectedFrom + "'");
		}

		return listDatasourcesUsingQuery("/*[" + swCond.toString() + "]");

	}

	@Override
	public synchronized boolean updateActivationStatus(final String dsId, final String ifaceId, final boolean active) throws DatasourceManagerServiceException {
		try {
			final Document prof = findProfile(dsId);

			final String profId = prof.valueOf("//RESOURCE_IDENTIFIER/@value");

			((Element) prof.selectSingleNode("//INTERFACE[@id='" + ifaceId + "']")).addAttribute("active", Boolean.toString(active));

			return registryLocator.getService().updateProfile(profId, prof.asXML(), REPOSITORY_SERVICE_RESOURCE_TYPE);
		} catch (Exception e) {
			log.error("Error updating profile " + dsId);
			throw new DatasourceManagerServiceException("Error updating profile " + dsId, e);
		}
	}

	@Override
	public synchronized boolean updateLevelOfCompliance(final String dsId, final String ifaceId, final String level) throws DatasourceManagerServiceException {
		try {
			final Document prof = findProfile(dsId);

			final String profId = prof.valueOf("//RESOURCE_IDENTIFIER/@value");

			((Element) prof.selectSingleNode("//INTERFACE[@id='" + ifaceId + "']")).addAttribute("compliance", level);

			return registryLocator.getService().updateProfile(profId, prof.asXML(), REPOSITORY_SERVICE_RESOURCE_TYPE);
		} catch (Exception e) {
			log.error("Error updating profile " + dsId);
			throw new DatasourceManagerServiceException("Error updating profile " + dsId, e);
		}
	}

	@Override
	public boolean updateBaseUrl(final String dsId, final String ifaceId, final String baseUrl) throws DatasourceManagerServiceException {
		try {
			try {
				final Document prof = findProfile(dsId);

				final String profId = prof.valueOf("//RESOURCE_IDENTIFIER/@value");

				prof.selectSingleNode("//INTERFACE[@id='" + ifaceId + "']/BASE_URL").setText(baseUrl);

				return registryLocator.getService().updateProfile(profId, prof.asXML(), REPOSITORY_SERVICE_RESOURCE_TYPE);
			} catch (Exception e) {
				log.error("Error updating profile " + dsId);
				throw new DatasourceManagerServiceException("Error updating profile " + dsId, e);
			}
		} catch (Exception e) {
			log.error("");
			throw new DatasourceManagerServiceException("", e);
		}

	}

	@Override
	public boolean updateContentDescription(final String dsId, final String ifaceId, final String desc) throws DatasourceManagerServiceException {
		try {
			final Document prof = findProfile(dsId);

			final String profId = prof.valueOf("//RESOURCE_IDENTIFIER/@value");

			((Element) prof.selectSingleNode("//INTERFACE[@id='" + ifaceId + "']")).addAttribute("contentDescription", desc);

			return registryLocator.getService().updateProfile(profId, prof.asXML(), REPOSITORY_SERVICE_RESOURCE_TYPE);
		} catch (Exception e) {
			log.error("Error updating profile " + dsId);
			throw new DatasourceManagerServiceException("Error updating profile " + dsId, e);
		}
	}

	@Override
	public boolean setIisProcessingWorkflow(final String dsId, final String ifaceId, final String wf) throws DatasourceManagerServiceException {
		return updateExtraField(dsId, ifaceId, "iis_processing_workflow", wf, false);
	}

	@Override
	public synchronized boolean addInterface(final String dsId, final IfaceDesc iface) throws DatasourceManagerServiceException {
		try {
			final Document prof = findProfile(dsId);

			final String profId = prof.valueOf("//RESOURCE_IDENTIFIER/@value");

			((Element) prof.selectSingleNode("//INTERFACES")).add(SimpleDatasourceMagerUtils.toInterfaceNode(iface));

			return registryLocator.getService().updateProfile(profId, prof.asXML(), REPOSITORY_SERVICE_RESOURCE_TYPE);
		} catch (Exception e) {
			log.error("Error adding an interface to ds " + dsId);
			throw new DatasourceManagerServiceException("Error adding an interface to ds " + dsId, e);
		}

	}

	@Override
	public synchronized boolean deleteInterface(final String dsId, final String ifcId) throws DatasourceManagerServiceException {
		try {
			final Document prof = findProfile(dsId);

			final String profId = prof.valueOf("//RESOURCE_IDENTIFIER/@value");

			prof.selectSingleNode("//INTERFACE[@id='" + ifcId + "']").detach();

			return registryLocator.getService().updateProfile(profId, prof.asXML(), REPOSITORY_SERVICE_RESOURCE_TYPE);
		} catch (Exception e) {
			log.error("");
			throw new DatasourceManagerServiceException("", e);
		}

	}

	@Override
	public synchronized boolean updateExtraField(final String dsId, final String ifaceId, final String field, final String value, final boolean preserveOriginal)
			throws DatasourceManagerServiceException {
		try {
			final Document prof = findProfile(dsId);

			final String profId = prof.valueOf("//RESOURCE_IDENTIFIER/@value");

			SimpleDatasourceMagerUtils.addOrupdateExtraField(prof, field, value);

			return registryLocator.getService().updateProfile(profId, prof.asXML(), REPOSITORY_SERVICE_RESOURCE_TYPE);
		} catch (Exception e) {
			log.error("Error updating extrafield in profile " + dsId);
			throw new DatasourceManagerServiceException("Error updating extrafield in profile " + dsId, e);
		}
	}

	@Override
	public synchronized boolean updateAccessParam(final String dsId,
			final String ifaceId,
			final String field,
			final String value,
			final boolean preserveOriginal) throws DatasourceManagerServiceException {
		try {
			final Document prof = findProfile(dsId);

			final String profId = prof.valueOf("//RESOURCE_IDENTIFIER/@value");

			final Element node = (Element) prof.selectSingleNode("//INTERFACE[@id='" + ifaceId + "']/ACCESS_PROTOCOL");
			if (node != null) {
				node.setText(value);
				node.addAttribute(field, value);
			}
			return registryLocator.getService().updateProfile(profId, prof.asXML(), REPOSITORY_SERVICE_RESOURCE_TYPE);
		} catch (Exception e) {
			log.error("Error updating accessparam in profile " + dsId);
			throw new DatasourceManagerServiceException("Error updating accessparam in profile " + dsId, e);
		}
	}

	@Override
	public synchronized boolean deleteAccessParamOrExtraField(final String dsId, final String ifaceId, final String field)
			throws DatasourceManagerServiceException {

		try {
			final Document prof = findProfile(dsId);

			final String profId = prof.valueOf("//RESOURCE_IDENTIFIER/@value");

			final Node nodeAP = prof.selectSingleNode("//INTERFACE[@id='" + ifaceId + "']/ACCESS_PROTOCOL/@" + field);
			if (nodeAP != null) {
				nodeAP.detach();
			}
			final Node nodeEF = prof.selectSingleNode("//INTERFACE[@id='" + ifaceId + "']/INTERFACE_EXTRA_FIELD[@name='" + field + "']");
			if (nodeEF != null) {
				nodeEF.detach();
			}

			return registryLocator.getService().updateProfile(profId, prof.asXML(), REPOSITORY_SERVICE_RESOURCE_TYPE);
		} catch (Exception e) {
			log.error("Error updating accessparam in profile " + dsId);
			throw new DatasourceManagerServiceException("Error updating accessparam in profile " + dsId, e);
		}

	}

	private Document findProfile(final String dsId) throws DatasourceManagerServiceException {
		try {
			final String profile = lookupLocator.getService().getResourceProfileByQuery("/*[.//DATASOURCE_ORIGINAL_ID  = '" + dsId + "']");

			return reader.read(new StringReader(profile));
		} catch (Exception e) {
			throw new DatasourceManagerServiceException("Error finding  profile " + dsId, e);
		}
	}

	private String findProfileId(final String dsId) throws DatasourceManagerServiceException {
		try {
			return lookupLocator.getService().getResourceProfileByQuery("/*[.//DATASOURCE_ORIGINAL_ID = '" + dsId + "']//RESOURCE_IDENTIFIER/@value/string()");
		} catch (Exception e) {
			throw new DatasourceManagerServiceException("Error finding id of profile " + dsId, e);
		}
	}

	public List<DatasourceDesc> listDatasourcesUsingQuery(final String xquery) throws DatasourceManagerServiceException {
		try {
			final List<DatasourceDesc> list = Lists.newArrayList();
			for (String prof : lookupLocator.getService().quickSearchProfile(xquery)) {
				list.add(SimpleDatasourceMagerUtils.toDatasourceDesc(reader.read(new StringReader(prof))));
			}
			return list;
		} catch (Exception e) {
			log.error("Error obtaining datasource list using query: " + xquery);
			throw new DatasourceManagerServiceException("Error obtaining datasource list", e);
		}
	}

	@Override
	public boolean updateSQL(final String dsId, final String sql, final boolean delete) throws DatasourceManagerServiceException {
		throw new DatasourceManagerServiceException("NOT IMPLEMENTED");
	}

	public ServiceLocator<ISRegistryService> getRegistryLocator() {
		return registryLocator;
	}

	@Required
	public void setRegistryLocator(final ServiceLocator<ISRegistryService> registryLocator) {
		this.registryLocator = registryLocator;
	}

	public ServiceLocator<ISLookUpService> getLookupLocator() {
		return lookupLocator;
	}

	@Required
	public void setLookupLocator(final ServiceLocator<ISLookUpService> lookupLocator) {
		this.lookupLocator = lookupLocator;
	}

}
