package eu.dnetlib.pace.config;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import eu.dnetlib.pace.clustering.Acronyms;
import eu.dnetlib.pace.clustering.Clustering;
import eu.dnetlib.pace.clustering.ClusteringFunction;
import eu.dnetlib.pace.clustering.NgramPairs;
import eu.dnetlib.pace.clustering.Ngrams;
import eu.dnetlib.pace.clustering.RandomClusteringFunction;
import eu.dnetlib.pace.clustering.SuffixPrefix;
import eu.dnetlib.pace.common.AbstractPaceFunctions;
import eu.dnetlib.pace.condition.AlwaysTrueCondition;
import eu.dnetlib.pace.condition.ConditionAlgo;
import eu.dnetlib.pace.condition.TitleVersionMatch;
import eu.dnetlib.pace.condition.YearMatch;
import eu.dnetlib.pace.distance.DistanceAlgo;
import eu.dnetlib.pace.distance.JaroWinkler;
import eu.dnetlib.pace.distance.JaroWinklerTitle;
import eu.dnetlib.pace.distance.Level2JaroWinkler;
import eu.dnetlib.pace.distance.Level2Levenstein;
import eu.dnetlib.pace.distance.Levenstein;
import eu.dnetlib.pace.distance.NullDistanceAlgo;
import eu.dnetlib.pace.distance.SubStringLevenstein;
import eu.dnetlib.pace.distance.YearLevenstein;
import eu.dnetlib.pace.model.ClusteringDef;
import eu.dnetlib.pace.model.CondDef;
import eu.dnetlib.pace.model.FieldDef;

public abstract class ConfigurableModel extends OverrideConfig {

	private Predicate<String> filterStrict = new Predicate<String>() {
		@Override
		public boolean apply(String key) {
			return !key.equals("strict");
		}
	};

	@Override
	public List<FieldDef> fields() {
		return parseFields("");
	}
	
	@Override
	public List<CondDef> conditions() {
		return parseConds("");
	}
	
	@Override
	public List<ClusteringDef> clusterings() {
		return parseClustering("");
	}
	
	@Override
	public Map<String, Set<String>> blacklists() {
		return parseBlacklists();
	}

	@Override
	public List<FieldDef> strictFields() {
		return parseFields(".strict");
	}
	
	@Override
	public FieldDef identifierFieldDef() {
		return new FieldDef(identifierField(), new NullDistanceAlgo(), false);
	}

	private List<FieldDef> parseFields(final String base) {
		@SuppressWarnings("unchecked")
		final Map<String, ?> modelMap = (Map<String, ?>) config.getObject("pace.conf.model");
		return Lists.newArrayList(Iterables.transform(filter(modelMap).entrySet(), new Function<Entry<String, ?>, FieldDef>() {
			@Override
			public FieldDef apply(Entry<String, ?> e) {

				final String name = e.getKey();

				double weight = config.getDouble(String.format("pace.conf.model%s.%s.weight", base, name));
				boolean ignoreMissing = config.getBoolean(String.format("pace.conf.model%s.%s.ignoreMissing", base, name));
				//Type type = Type.valueOf(config.getString(String.format("pace.conf.model%s.%s.type", base, name)));
				
				return new FieldDef(name, getAlgo(base, name, weight), ignoreMissing);
			}

			private DistanceAlgo getAlgo(final String base, String name, double w) {
				switch (Algo.valueOf(config.getString(String.format("pace.conf.model%s.%s.algo", base, name)))) {
				case JaroWinkler:
					return new JaroWinkler(w);
				case JaroWinklerTitle:
					return new JaroWinklerTitle(w);
				case Level2JaroWinkler:
					return new Level2JaroWinkler(w);
				case Level2Levenstein:
					return new Level2Levenstein(w);
				case Levenstein:
					return new Levenstein(w);
				case SubStringLevenstein:
					return new SubStringLevenstein(w, config.getInt(String.format("pace.conf.model%s.%s.limit", base, name)));
				case YearLevenstein:
					return new YearLevenstein(w, config.getInt(String.format("pace.conf.model%s.%s.limit", base, name)));					
				case Null:
					return new NullDistanceAlgo();
				default:
					return new NullDistanceAlgo();
				}
			}
		}));
	}
	
	public List<CondDef> parseConds(final String base) {
		@SuppressWarnings("unchecked")
		final Map<String, ?> modelMap = (Map<String, ?>) config.getObject("pace.conf.conditions");
		return Lists.newArrayList(Iterables.transform(filter(modelMap).entrySet(), new Function<Entry<String, ?>, CondDef>() {
			@Override
			public CondDef apply(Entry<String, ?> e) {

				final Cond condName = Cond.valueOf(e.getKey());
				final List<String> fields = config.getList(String.format("pace.conf.conditions%s.%s.fields", base, e.getKey()));
				
				return new CondDef(getCondAlgo(fields, condName));
			}

			private ConditionAlgo getCondAlgo(List<String> fields, Cond condName) {
				switch (condName) {
				case yearMatch:
					return new YearMatch(fields);
				case titleVersionMatch:
					return new TitleVersionMatch(fields);
				default:
					return new AlwaysTrueCondition(fields);
				}
			}
		}));
	}
	
	public List<ClusteringDef> parseClustering(final String base) {
		@SuppressWarnings("unchecked")
		final Map<String, ?> modelMap = (Map<String, ?>) config.getObject("pace.conf.clustering");
		return Lists.newArrayList(Iterables.transform(filter(modelMap).entrySet(), new Function<Entry<String, ?>, ClusteringDef>() {
			@Override
			public ClusteringDef apply(Entry<String, ?> e) {

				final Clustering clustering = Clustering.valueOf(e.getKey());
				final List<String> fields = config.getList(String.format("pace.conf.clustering%s.%s.fields", base, e.getKey()));
				@SuppressWarnings("unchecked")
				final Map<String, Integer> params = (Map<String, Integer>) config.getObject(String.format("pace.conf.clustering%s.%s.params", base, e.getKey()));
				
				return new ClusteringDef(clustering, getClusteringFunction(params, clustering), fields);
			}

			private ClusteringFunction getClusteringFunction(Map<String, Integer> params, Clustering clustering) {
				switch (clustering) {
				case acronyms:
					return new Acronyms(params);
				case ngrams:
					return new Ngrams(params);
				case ngrampairs:
					return new NgramPairs(params);
				case suffixprefix:
					return new SuffixPrefix(params);
				default:
					return new RandomClusteringFunction(params);
				}
			}
		}));
	}
	
	@SuppressWarnings("unchecked")
	public Map<String, Set<String>> parseBlacklists() {
		Map<String, Set<String>> res = Maps.newHashMap();
		for(String field : (List<String>) (config.hasPath("pace.conf.blacklists") ? config.getList("pace.conf.blacklists") : new ArrayList<String>())) {
			res.put(field, AbstractPaceFunctions.loadFromClasspath("/eu/dnetlib/pace/config/" + field + "_blacklist.txt"));
		}
		return res;
	}

	private Map<String, ?> filter(final Map<String, ?> modelMap) {
		return modelMap != null ? Maps.filterKeys(modelMap, filterStrict) : new HashMap<String, Object>();
	}	

}
