package eu.dnetlib.msro.openaireplus.workflows.nodes.hbase;

import java.io.IOException;
import java.util.List;

import javax.xml.ws.wsaddressing.W3CEndpointReference;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.util.Bytes;
import org.springframework.beans.factory.annotation.Required;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.NodeToken;

import eu.dnetlib.data.transform.Column;
import eu.dnetlib.data.transform.Row;
import eu.dnetlib.data.transform.XsltRowTransformer;
import eu.dnetlib.data.transform.XsltRowTransformerFactory;
import eu.dnetlib.enabling.resultset.client.ResultSetClientFactory;
import eu.dnetlib.msro.workflows.nodes.AsyncJobNode;
import eu.dnetlib.msro.workflows.nodes.ProgressJobNode;
import eu.dnetlib.msro.workflows.resultset.ProcessCountingResultSetFactory;
import eu.dnetlib.msro.workflows.util.ProgressProvider;
import eu.dnetlib.msro.workflows.util.ResultsetProgressProvider;
import eu.dnetlib.msro.workflows.util.WorkflowsConstants;

public class StoreHBaseRecordsJobNode extends AsyncJobNode implements ProgressJobNode {

	private static final Log log = LogFactory.getLog(StoreHBaseRecordsJobNode.class); // NOPMD by marko on 11/24/08 5:02 PM

	private ResultSetClientFactory resultSetClientFactory;
	private XsltRowTransformerFactory xsltRowTransformerFactory;

	private int batchSize;

	private String inputEprParam;
	private String hbaseConfParam;
	private String hbaseTableParam;
	private String xsltPath;

	private ProgressProvider progressProvider;

	private ProcessCountingResultSetFactory processCountingResultSetFactory;

	@Override
	protected String execute(final NodeToken token) throws Exception {

		final String epr = token.getEnv().getAttribute(inputEprParam);

		final ResultsetProgressProvider resultsetProgressProvider = processCountingResultSetFactory.createProgressProvider(token.getProcess(), epr);

		setProgressProvider(resultsetProgressProvider);

		doWrite(resultsetProgressProvider.getEpr(), token, getTransformer());

		log.info("finished import to HBase");

		return Arc.DEFAULT_ARC;
	}

	private XsltRowTransformer getTransformer() throws IOException {
		final String xslt = ((xsltPath == null) || xsltPath.isEmpty()) ? "" : IOUtils.toString(getClass().getResourceAsStream(xsltPath));
		return getXsltRowTransformerFactory().getTransformer(xslt);
	}

	private void doWrite(final W3CEndpointReference epr, final NodeToken token, final XsltRowTransformer transformer) throws IOException {
		final String tableName = token.getEnv().getAttribute(hbaseTableParam);
		final Configuration conf = (Configuration) token.getEnv().getTransientAttribute(hbaseConfParam);

		final HTable table = new HTable(conf, tableName);

		final List<Put> buffer = Lists.newArrayList();
		int count = 0;
		int nPuts = 0;
		try {
			for (Row row : Iterables.concat(Iterables.transform(getResultSetClientFactory().getClient(epr), transformer))) {
				log.debug(row);

				final Put put = new Put(Bytes.toBytes(row.getKey()));

				for (Column<String, byte[]> col : row) {
					put.add(Bytes.toBytes(row.getColumnFamily()), Bytes.toBytes(col.getName()), col.getValue());
					nPuts++;
				}

				buffer.add(put);

				if ((++count % getBatchSize()) == 0) {
					table.put(buffer);
					buffer.clear();
				}
			}
		} finally {
			if (!buffer.isEmpty()) {
				table.put(buffer);
			}
			table.flushCommits();
			table.close();

			token.getEnv().setAttribute(WorkflowsConstants.MAIN_LOG_PREFIX + "N_ROWS", count);
			token.getEnv().setAttribute(WorkflowsConstants.MAIN_LOG_PREFIX + "N_PUTS", nPuts);
		}
	}

	public ResultSetClientFactory getResultSetClientFactory() {
		return resultSetClientFactory;
	}

	@Required
	public void setResultSetClientFactory(final ResultSetClientFactory resultSetClientFactory) {
		this.resultSetClientFactory = resultSetClientFactory;
	}

	public int getBatchSize() {
		return batchSize;
	}

	@Required
	public void setBatchSize(final int batchSize) {
		this.batchSize = batchSize;
	}

	public String getInputEprParam() {
		return inputEprParam;
	}

	public void setInputEprParam(final String inputEprParam) {
		this.inputEprParam = inputEprParam;
	}

	public String getHbaseConfParam() {
		return hbaseConfParam;
	}

	public void setHbaseConfParam(final String hbaseConfParam) {
		this.hbaseConfParam = hbaseConfParam;
	}

	public String getHbaseTableParam() {
		return hbaseTableParam;
	}

	public void setHbaseTableParam(final String hbaseTableParam) {
		this.hbaseTableParam = hbaseTableParam;
	}

	@Required
	public void setProcessCountingResultSetFactory(final ProcessCountingResultSetFactory processCountingResultSetFactory) {
		this.processCountingResultSetFactory = processCountingResultSetFactory;
	}

	@Override
	public ProgressProvider getProgressProvider() {
		return progressProvider;
	}

	public XsltRowTransformerFactory getXsltRowTransformerFactory() {
		return xsltRowTransformerFactory;
	}

	@Required
	public void setXsltRowTransformerFactory(final XsltRowTransformerFactory xsltRowTransformerFactory) {
		this.xsltRowTransformerFactory = xsltRowTransformerFactory;
	}

	public void setProgressProvider(final ProgressProvider progressProvider) {
		this.progressProvider = progressProvider;
	}

	public ProcessCountingResultSetFactory getProcessCountingResultSetFactory() {
		return processCountingResultSetFactory;
	}

	public String getXsltPath() {
		return xsltPath;
	}

	public void setXsltPath(final String xsltPath) {
		this.xsltPath = xsltPath;
	}

}
