package eu.dnetlib.msro.openaireplus.workflows.nodes.hbase;

import java.io.IOException;
import java.util.List;
import java.util.Map.Entry;
import java.util.Set;

import javax.annotation.Resource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.client.HBaseAdmin;
import org.apache.hadoop.hbase.util.Bytes;
import org.springframework.beans.factory.annotation.Required;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.NodeToken;

import eu.dnetlib.data.hadoop.config.ClusterName;
import eu.dnetlib.data.hadoop.config.ConfigurationEnumerator;
import eu.dnetlib.data.mapreduce.hbase.HBaseTableUtils;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;
import eu.dnetlib.msro.workflows.util.WorkflowsConstants;

public class PrepareHBaseJobNode extends SimpleJobNode {

	@Resource
	private ConfigurationEnumerator configurationEnumerator;

	private String hbaseConfParam;

	private String dbParam;
	private String db;

	private String hbaseTableParam;
	private String hbaseTable;

	private static final Log log = LogFactory.getLog(PrepareHBaseJobNode.class); // NOPMD by marko on 11/24/08 5:02 PM

	private final Function<HColumnDescriptor, String> columnName = new Function<HColumnDescriptor, String>() {

		@Override
		public String apply(final HColumnDescriptor d) {
			return d.getNameAsString();
		}
	};

	@Override
	protected String execute(final NodeToken token) throws Exception {

		final Configuration hbaseConf = configurationEnumerator.get(ClusterName.DM);
		if (log.isDebugEnabled()) {
			for (Entry<String, String> e : hbaseConf) {
				token.getEnv().setAttribute("DEBUG:" + e.getKey(), e.getValue());
			}
		}

		token.getEnv().setAttribute(dbParam, db);
		token.getEnv().setAttribute(hbaseTableParam, hbaseTable);

		token.getEnv().setAttribute(WorkflowsConstants.MAIN_LOG_PREFIX + dbParam, db);
		token.getEnv().setAttribute(WorkflowsConstants.MAIN_LOG_PREFIX + hbaseTableParam, hbaseTable);

		token.getEnv().setTransientAttribute(hbaseConfParam, hbaseConf);

		checkTable(hbaseConf);

		return Arc.DEFAULT_ARC;
	}

	private void checkTable(final Configuration conf) throws IOException {

		final HBaseAdmin admin = new HBaseAdmin(conf);
		try {
			if (!admin.tableExists(getHbaseTable())) {
				log.info("cannot find hbase table: " + getHbaseTable());
				createTable(admin, getHbaseTable());
			}

			final HTableDescriptor desc = admin.getTableDescriptor(Bytes.toBytes(getHbaseTable()));

			final Set<String> foundColumns = Sets.newHashSet(Iterables.transform(Lists.newArrayList(desc.getColumnFamilies()), columnName));

			Set<String> allColumns = HBaseTableUtils.listColumns();
			log.info("ensuring columns on table " + getHbaseTable() + ": " + allColumns);
			final List<String> missingColumns = Lists.newArrayList(Iterables.filter(allColumns, new Predicate<String>() {

				@Override
				public boolean apply(final String name) {
					return !foundColumns.contains(name);
				}
			}));

			if (!missingColumns.isEmpty()) {

				if (admin.isTableEnabled(getHbaseTable())) {
					admin.disableTable(getHbaseTable());
				}

				for (String column : missingColumns) {
					log.info("hbase table: '" + getHbaseTable() + "', adding column: " + column);
					admin.addColumn(getHbaseTable(), new HColumnDescriptor(column));
				}

				admin.enableTable(getHbaseTable());
			}
		} finally {
			admin.close();
		}
	}

	private void createTable(final HBaseAdmin admin, final String tableName) throws IOException {
		if (admin.tableExists(tableName)) {
			admin.disableTable(tableName);
			admin.deleteTable(tableName);
		}

		HTableDescriptor desc = new HTableDescriptor(tableName);
		for (String column : HBaseTableUtils.listColumns()) {
			HColumnDescriptor hds = new HColumnDescriptor(column);
			hds.setMaxVersions(1);
			desc.addFamily(hds);
		}
		log.info("creating hbase table: " + tableName);

		admin.createTable(desc);
		log.info("created hbase table: [" + tableName + "] descriptor: [" + desc.toString() + "]");
	}

	public String getDbParam() {
		return dbParam;
	}

	public void setDbParam(final String dbParam) {
		this.dbParam = dbParam;
	}

	public String getDb() {
		return db;
	}

	@Required
	public void setDb(final String db) {
		this.db = db;
	}

	public String getHbaseConfParam() {
		return hbaseConfParam;
	}

	public void setHbaseConfParam(final String hbaseConfParam) {
		this.hbaseConfParam = hbaseConfParam;
	}

	public String getHbaseTableParam() {
		return hbaseTableParam;
	}

	public void setHbaseTableParam(final String hbaseTableParam) {
		this.hbaseTableParam = hbaseTableParam;
	}

	public String getHbaseTable() {
		return hbaseTable;
	}

	@Required
	public void setHbaseTable(final String hbaseTable) {
		this.hbaseTable = hbaseTable;
	}
}
