package eu.dnetlib.msro.openaireplus.workflows.nodes.hbase;

import java.io.File;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;

import javax.annotation.Resource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.mapred.JobConf;
import org.apache.hadoop.mapred.RunningJob;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import com.google.gson.Gson;
import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.NodeToken;

import eu.dnetlib.data.hadoop.config.ClusterName;
import eu.dnetlib.data.mapreduce.HadoopJob;
import eu.dnetlib.data.mapreduce.JobClientResolver;
import eu.dnetlib.data.mapreduce.JobParams;
import eu.dnetlib.msro.workflows.nodes.ProgressJobNode;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;
import eu.dnetlib.msro.workflows.util.MsroPropertyFetcher;
import eu.dnetlib.msro.workflows.util.ProgressProvider;

public class HBaseMapReduceJobNode extends SimpleJobNode implements ProgressJobNode, ApplicationContextAware {

	private static final Log log = LogFactory.getLog(HBaseMapReduceJobNode.class);

	@Resource
	private JobClientResolver jobClientResolver;

	private MsroPropertyFetcher propertyFetcher;

	private ProgressProvider progressProvider;

	private String hadoopJob;
	private String sourceTableProp;
	private String targetTableProp;
	private String properties;

	private ApplicationContext context;

	@Override
	protected String execute(final NodeToken token) throws Exception {
		final JobConf jobConf = getJobConf(token, ClusterName.DM);

		token.getEnv().setAttribute("mapreduce.map.class", jobConf.get("mapreduce.map.class", ""));
		token.getEnv().setAttribute("mapreduce.reduce.class", jobConf.get("mapreduce.reduce.class", ""));

		log.info("submitting job " + jobConf.getJobName());

		// add the client side job parameters to the env for debugging
		for (Entry<String, String> e : jobConf) {
			log.debug(e.getKey() + ": " + e.getValue());
			token.getEnv().setAttribute(e.getKey(), e.getValue());
		}

		final RunningJob handle = jobClientResolver.getClient(ClusterName.DM).submitJob(jobConf);

		log.info("submitted job " + jobConf.getJobName() + " handle: " + handle.toString());

		setProgressProvider(new MapreduceProgressProvider(handle));

		token.getEnv().setTransientAttribute("hadoopConf", jobConf);
		token.getEnv().setTransientAttribute("runningJob", handle);

		handle.waitForCompletion();

		if (!handle.isSuccessful()) { throw new RuntimeException(handle.getTrackingURL() + "\n" + handle.getFailureInfo()); }
		return Arc.DEFAULT_ARC;
	}

	private JobConf getJobConf(final NodeToken token, final ClusterName name) {
		final Properties prop = new Properties();
		prop.setProperty(JobParams.HBASE_SOURCE_TABLE, propertyFetcher.getProps().getProperty(getSourceTableProp()));
		prop.setProperty(JobParams.HBASE_TARGET_TABLE, propertyFetcher.getProps().getProperty(getTargetTableProp()));
		if ((properties != null) && !properties.isEmpty()) {
			Map<?, ?> map = (new Gson()).fromJson(properties, Map.class);
			for (Entry<?, ?> e : map.entrySet()) {
				prop.setProperty(e.getKey().toString(), e.getValue().toString());
			}
		}

		final HadoopJob job = (HadoopJob) context.getBean(hadoopJob, HadoopJob.class);

		final JobConf jobConf = new JobConf(job.setJobDetails(name, prop).getConfiguration());

		final String libPath = token.getEnv().getAttribute("job.lib");
		if ((libPath == null) || !new File(libPath).canRead()) { throw new IllegalArgumentException("unable to read mapreduce library: " + libPath); }

		jobConf.setJobName(job.getName());
		jobConf.setJar(libPath);

		return jobConf;
	}

	protected void log(final JobConf jobConf) {
		for (final Entry<String, String> e : jobConf) {
			log.debug(e.getKey() + " " + e.getValue());
		}
	}

	public String getHadoopJob() {
		return hadoopJob;
	}

	public void setHadoopJob(final String hadoopJob) {
		this.hadoopJob = hadoopJob;
	}

	@Override
	public ProgressProvider getProgressProvider() {
		return progressProvider;
	}

	public void setProgressProvider(final ProgressProvider progressProvider) {
		this.progressProvider = progressProvider;
	}

	public String getProperties() {
		return properties;
	}

	public void setProperties(final String properties) {
		this.properties = properties;
	}

	@Override
	public void setApplicationContext(final ApplicationContext context) throws BeansException {
		this.context = context;
	}

	public String getSourceTableProp() {
		return sourceTableProp;
	}

	public void setSourceTableProp(final String sourceTableProp) {
		this.sourceTableProp = sourceTableProp;
	}

	public String getTargetTableProp() {
		return targetTableProp;
	}

	public void setTargetTableProp(final String targetTableProp) {
		this.targetTableProp = targetTableProp;
	}

	public MsroPropertyFetcher getPropertyFetcher() {
		return propertyFetcher;
	}

	@Required
	public void setPropertyFetcher(final MsroPropertyFetcher propertyFetcher) {
		this.propertyFetcher = propertyFetcher;
	}

}
