package eu.dnetlib.functionality.modular.ui.workflows.util;

import java.io.StringReader;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.xml.transform.stream.StreamSource;

import org.apache.commons.lang.math.NumberUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;

import com.google.common.base.Function;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import eu.dnetlib.functionality.modular.ui.workflows.objects.AdvancedMetaWorkflowDescriptor;
import eu.dnetlib.functionality.modular.ui.workflows.objects.AtomicWorkflowDescriptor;
import eu.dnetlib.functionality.modular.ui.workflows.objects.DatasourceWorkflowDescriptor;
import eu.dnetlib.functionality.modular.ui.workflows.objects.MetaWorkflowDescriptor;
import eu.dnetlib.functionality.modular.ui.workflows.objects.NodeInfo;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.miscutils.functional.xml.ApplyXslt;
import eu.dnetlib.msro.workflows.util.WorkflowParam;
import eu.dnetlib.msro.workflows.util.WorkflowsConstants;
import eu.dnetlib.msro.workflows.util.WorkflowsConstants.WorkflowStatus;

public class TransformerUtils {

	private static final Log log = LogFactory.getLog(TransformerUtils.class);

	public static final ApplyXslt toSarasvatiMetaWorkflow = new ApplyXslt(new StreamSource(
			TransformerUtils.class.getResourceAsStream("/eu/dnetlib/msro/workflows/xslt/metawf_profile2sarasvati.xslt")));

	public static final Function<String, DatasourceWorkflowDescriptor> toDatasourceWorkflow = new Function<String, DatasourceWorkflowDescriptor>() {

		@Override
		public DatasourceWorkflowDescriptor apply(final String s) {
			try {
				final SAXReader reader = new SAXReader();
				final Document doc = reader.read(new StringReader(s));

				final String dsId = doc.valueOf("//dsId");
				final String name = doc.valueOf("//dsName");
				final Set<String> dsProtocols = Sets.newHashSet();
				for (Object o : doc.selectNodes("//protocols/ACCESS_PROTOCOL")) {
					dsProtocols.add(((Node) o).getText());
				}
				final String dsTypology = doc.valueOf("//dsType");
				final Set<MetaWorkflowDescriptor> metaworkflows = Sets.newHashSet();
				for (Object o : doc.selectNodes("//metaworkflows/workflow")) {
					final String wfId = ((Node) o).valueOf("@id");
					final String wfName = ((Node) o).valueOf("@name");
					final String wfStatusTemp = ((Node) o).valueOf("@status");
					final String wfFamily = ((Node) o).valueOf("@family");
					final WorkflowStatus wfStatus = ((wfStatusTemp == null) || wfStatusTemp.isEmpty()) ? WorkflowStatus.MISSING : WorkflowStatus
							.valueOf(wfStatusTemp);
					metaworkflows.add(new MetaWorkflowDescriptor(wfId, wfName, wfStatus, wfFamily));
				}
				return new DatasourceWorkflowDescriptor(dsId, name, dsProtocols, dsTypology, metaworkflows);
			} catch (Exception e) {
				log.error("Error building DatasourceWorkflowDescriptor form xml: " + s, e);
				throw new IllegalArgumentException(e);
			}
		}
	};

	public static final Function<String, MetaWorkflowDescriptor> toMetaWorkflow = new Function<String, MetaWorkflowDescriptor>() {

		@Override
		public MetaWorkflowDescriptor apply(final String s) {
			Iterator<String> iter = Splitter.on("@@@").trimResults().split(s).iterator();
			String id = iter.next();
			String name = iter.next();
			String family = iter.next();
			WorkflowStatus status = WorkflowStatus.valueOf(iter.next());
			return new MetaWorkflowDescriptor(id, name, status, family);
		}
	};

	public static final Function<String, NodeInfo> toNodeInfo = new Function<String, NodeInfo>() {

		@Override
		public NodeInfo apply(final String xml) {
			try {
				final SAXReader reader = new SAXReader();
				final Document doc = reader.read(new StringReader(xml));
				final List<WorkflowParam> userParams = Lists.newArrayList();
				final List<WorkflowParam> systemParams = Lists.newArrayList();

				for (Object o : doc.selectNodes("//PARAM")) {
					Node n = (Node) o;
					String key = n.valueOf("@name");
					boolean required = "true".equalsIgnoreCase(n.valueOf("@required"));
					String type = n.valueOf("@type");
					String function = n.valueOf("@function");
					String value = n.getText();
					if ("user".equals(n.valueOf("@managedBy"))) {
						userParams.add(new WorkflowParam(key, value, required, type, function));
					} else {
						systemParams.add(new WorkflowParam(key, value, required, type, function));
					}
				}
				String name = doc.valueOf("/NODE/@name");
				String desc = doc.valueOf("//DESCRIPTION");

				return new NodeInfo(name, desc, userParams, systemParams);
			} catch (Exception e) {
				log.error("Error parsing xml: " + xml, e);
			}
			return new NodeInfo("", "", new ArrayList<WorkflowParam>(), new ArrayList<WorkflowParam>());
		}
	};

	public static final Function<String, AtomicWorkflowDescriptor> toAtomicWorkflow = new Function<String, AtomicWorkflowDescriptor>() {

		@Override
		public AtomicWorkflowDescriptor apply(final String xml) {
			try {
				final SAXReader reader = new SAXReader();
				final Document doc = reader.read(new StringReader(xml));

				final String id = doc.valueOf("//RESOURCE_IDENTIFIER/@value");
				final String name = doc.valueOf("//WORKFLOW_NAME");
				final String imageUrl = "wf_atomic_workflow.img?id=" + id + "&t=" + DateUtils.now();
				final String lastExecutionDate = doc.valueOf("//STATUS/LAST_EXECUTION_DATE");
				final String start = doc.valueOf("//CONFIGURATION/@start");
				final boolean ready = doc.selectNodes("//PARAM[@required='true' and string-length(normalize-space(.)) = 0]").isEmpty();
				final int priority = NumberUtils.toInt(doc.valueOf("//WORKFLOW_PRIORITY"), WorkflowsConstants.DEFAULT_WF_PRIORITY);
				return new AtomicWorkflowDescriptor(id, name, priority, "", null, imageUrl, ready, start, lastExecutionDate);
			} catch (Exception e) {
				log.error("Error parsing xml: " + xml, e);
			}
			return new AtomicWorkflowDescriptor("", "", 0, "", null, "", false, "disabled", "");
		}
	};

	public static final Function<String, AdvancedMetaWorkflowDescriptor> toAdvancedMetaWorkflow = new Function<String, AdvancedMetaWorkflowDescriptor>() {

		@Override
		public AdvancedMetaWorkflowDescriptor apply(final String xml) {
			try {
				final SAXReader reader = new SAXReader();
				final Document doc = reader.read(new StringReader(xml));

				final String id = doc.valueOf("//RESOURCE_IDENTIFIER/@value");
				final String name = doc.valueOf("//METAWORKFLOW_NAME");
				final String family = doc.valueOf("//METAWORKFLOW_NAME/@family");
				final String mapContent = null;

				final String imageUrl = "wf_metaworkflow.img?id=" + id + "&t=" + DateUtils.now();
				final String email = doc.valueOf("//ADMIN_EMAIL");

				final boolean scheduled = Boolean.valueOf(doc.valueOf("//SCHEDULING/@enabled"));;
				final String cronExpression = doc.valueOf("//SCHEDULING/CRON");;
				final int minInterval = Integer.parseInt(doc.valueOf("//SCHEDULING/MININTERVAL"));
				final WorkflowStatus status = WorkflowStatus.valueOf(doc.valueOf("//CONFIGURATION/@status"));

				return new AdvancedMetaWorkflowDescriptor(id, name, mapContent, imageUrl, email, scheduled, cronExpression, minInterval, status, family);
			} catch (Exception e) {
				log.error("Error parsing xml: " + xml, e);
			}
			return new AdvancedMetaWorkflowDescriptor("", "", "", "", "", false, "", 0, WorkflowStatus.MISSING, "");
		}
	};
}
