package eu.dnetlib.functionality.modular.ui.workflows.controllers;

import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.OutputStream;
import java.io.StringReader;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Resource;
import javax.imageio.ImageIO;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gson.Gson;
import com.googlecode.sarasvati.GraphProcess;
import com.googlecode.sarasvati.Node;
import com.googlecode.sarasvati.NodeToken;
import com.googlecode.sarasvati.ProcessState;

import edu.emory.mathcs.backport.java.util.Collections;
import eu.dnetlib.common.logging.DnetLogger;
import eu.dnetlib.common.logging.LogMessage;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.functionality.modular.ui.workflows.objects.AdvancedMetaWorkflowDescriptor;
import eu.dnetlib.functionality.modular.ui.workflows.objects.AtomicWorkflowDescriptor;
import eu.dnetlib.functionality.modular.ui.workflows.objects.NodeInfo;
import eu.dnetlib.functionality.modular.ui.workflows.objects.NodeTokenInfo;
import eu.dnetlib.functionality.modular.ui.workflows.objects.ProcessListEntry;
import eu.dnetlib.functionality.modular.ui.workflows.objects.sections.AbstractWorkflowSection;
import eu.dnetlib.functionality.modular.ui.workflows.objects.sections.DataproviderWorkflowSection;
import eu.dnetlib.functionality.modular.ui.workflows.objects.sections.WorkflowSectionGrouper;
import eu.dnetlib.functionality.modular.ui.workflows.sarasvati.viewer.ProcessGraphGenerator;
import eu.dnetlib.functionality.modular.ui.workflows.util.ISLookupClient;
import eu.dnetlib.functionality.modular.ui.workflows.util.ISRegistryClient;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.msro.workflows.sarasvati.loader.ProfileToSarasvatiConverter;
import eu.dnetlib.msro.workflows.sarasvati.loader.WorkflowExecutor;
import eu.dnetlib.msro.workflows.sarasvati.registry.GraphProcessRegistry;
import eu.dnetlib.msro.workflows.util.WorkflowsConstants;
import eu.dnetlib.msro.workflows.util.WorkflowsConstants.WorkflowStatus;

/**
 * Web controller for the UI
 * 
 * @author Michele Artini
 */

@Controller
public class WorkflowsController {

	private final class JournalEntryFunction implements Function<Map<String, String>, Map<String, String>> {

		@Override
		public Map<String, String> apply(final Map<String, String> input) {
			String wfName = input.get(WorkflowsConstants.SYSTEM_WF_PROFILE_NAME);
			if (input.containsKey(WorkflowsConstants.DATAPROVIDER_NAME)) {
				wfName += " (" + input.get(WorkflowsConstants.DATAPROVIDER_NAME) + ")";
			}
			final Map<String, String> map = Maps.newHashMap();
			map.put("procId", input.get(WorkflowsConstants.SYSTEM_WF_PROCESS_ID));
			map.put("wfId", input.get(WorkflowsConstants.SYSTEM_WF_PROFILE_ID));
			map.put("wfName", wfName);
			map.put("wfFamily", input.get(WorkflowsConstants.SYSTEM_WF_PROFILE_FAMILY));
			map.put("date", input.get(LogMessage.LOG_DATE_FIELD));
			map.put("prettyDate", input.get(LogMessage.LOG_PRETTY_PRINT_DATE));
			map.put("wfStatus", Boolean.valueOf(input.get(WorkflowsConstants.SYSTEM_COMPLETED_SUCCESSFULLY)) ? "SUCCESS" : "FAILURE");
			return map;
		}
	}

	@Resource
	private ISLookupClient isLookupClient;

	@Resource
	private ISRegistryClient isRegistryClient;

	@Resource
	private GraphProcessRegistry graphProcessRegistry;

	@Resource
	private ProcessGraphGenerator processGraphGenerator;

	@Resource
	private WorkflowSectionGrouper workflowSectionGrouper;

	@Resource
	private WorkflowExecutor workflowExecutor;

	@Resource
	private ProfileToSarasvatiConverter profileToSarasvatiConverter;

	@Resource(name = "msroWorkflowLogger")
	private DnetLogger dnetLogger;

	private static final Log log = LogFactory.getLog(WorkflowsController.class);

	@RequestMapping("/ui/wf_metaworkflowsForSection.json")
	public void getMetaWorkflowsForSection(final HttpServletResponse response, @RequestParam(value = "section", required = true) final String sectionName)
			throws ISLookUpException, IOException {

		final AbstractWorkflowSection section = workflowSectionGrouper.getSectionByName(sectionName);
		String json = (section == null) ? "[]" : new Gson().toJson(section.listWorkflows());

		IOUtils.copy(new StringReader(json), response.getOutputStream());
	}

	@RequestMapping("/ui/wf_dataproviders.json")
	public void listDataproviders(final HttpServletResponse response) throws ISLookUpException, IOException {

		final DataproviderWorkflowSection section = workflowSectionGrouper.getDataproviderWorkflowSection();
		String json = (section == null) ? "[]" : new Gson().toJson(section.listWorkflows());

		IOUtils.copy(new StringReader(json), response.getOutputStream());
	}

	@RequestMapping("/ui/wf_metaworkflow.json")
	public void getMetaWorkflow(final HttpServletResponse response, @RequestParam(value = "id", required = true) final String id) throws Exception {
		String xml = isLookupClient.getSarasvatiMetaWorkflow(id);

		AdvancedMetaWorkflowDescriptor wf = isLookupClient.getMetaWorkflow(id);
		wf.setMapContent(processGraphGenerator.getMetaWfDescImageMap(id, xml));

		IOUtils.copy(new StringReader(new Gson().toJson(wf)), response.getOutputStream());
	}

	@RequestMapping("/ui/wf_atomic_workflow.json")
	public void getAtomicWorkflow(final HttpServletResponse response, @RequestParam(value = "id", required = true) final String id) throws Exception {
		String xml = profileToSarasvatiConverter.getSarasvatiWorkflow(id).getWorkflowXml();

		AtomicWorkflowDescriptor wf = isLookupClient.getAtomicWorkflow(id);
		wf.setMapContent(processGraphGenerator.getWfDescImageMap(id, xml));

		IOUtils.copy(new StringReader(new Gson().toJson(wf)), response.getOutputStream());
	}

	@RequestMapping("/ui/wf_atomic_workflow.img")
	public void showAtomicWorkflow(final HttpServletResponse response, @RequestParam(value = "id", required = true) final String id) throws Exception {

		String xml = profileToSarasvatiConverter.getSarasvatiWorkflow(id).getWorkflowXml();
		Set<String> notConfiguredNodes = isLookupClient.getNotConfiguredNodes(id);
		BufferedImage image = processGraphGenerator.getWfDescImage(id, xml, notConfiguredNodes);
		sendImage(response, image);
	}

	private void sendImage(final HttpServletResponse response, final BufferedImage image) throws IOException {
		response.setContentType("image/png");
		OutputStream out = response.getOutputStream();
		ImageIO.write(image, "png", out);
		out.flush();
		out.close();
	}

	@RequestMapping("/ui/wf.start")
	public void startWorkflow(final HttpServletResponse response, @RequestParam(value = "id", required = true) final String id) throws Exception {
		final String procId = workflowExecutor.startProcess(id);
		IOUtils.copy(new StringReader(new Gson().toJson(procId)), response.getOutputStream());
	}

	@RequestMapping("/ui/metawf.start")
	public void startMetaWorkflow(final HttpServletResponse response, @RequestParam(value = "id", required = true) final String id) throws Exception {
		workflowExecutor.startMetaWorkflow(id);
		IOUtils.copy(new StringReader(new Gson().toJson(id)), response.getOutputStream());
	}

	@RequestMapping("/ui/wf_workflow_node.json")
	public void workflowNode_info(final HttpServletResponse response,
			@RequestParam(value = "wf", required = true) final String wfId,
			@RequestParam(value = "node", required = true) final String nodeName) throws ISLookUpException, IOException {

		NodeInfo info = isLookupClient.getNodeInfo(wfId, nodeName);
		IOUtils.copy(new StringReader(new Gson().toJson(info)), response.getOutputStream());
	}

	@RequestMapping("/ui/wf_workflow_node.save")
	public void workflowNode_save(final HttpServletResponse response,
			@RequestParam(value = "wf", required = true) final String wfId,
			@RequestParam(value = "json", required = true) final String json) throws Exception {

		final Gson gson = new Gson();
		final NodeInfo info = gson.fromJson(json, NodeInfo.class);

		log.info("Updating node " + info.getName() + " of wf " + wfId);

		final String xml = isLookupClient.getProfile(wfId);
		boolean res = isRegistryClient.updateSarasvatiWorkflow(wfId, xml, info);

		for (String metaWfId : isLookupClient.listMetaWorflowsForWfId(wfId)) {
			if (isLookupClient.isExecutable(metaWfId)) {
				isRegistryClient.updateMetaWorkflowStatus(metaWfId, WorkflowStatus.EXECUTABLE);
			} else {
				isRegistryClient.updateMetaWorkflowStatus(metaWfId, WorkflowStatus.WAIT_USER_SETTINGS);
			}
		}

		IOUtils.copy(new StringReader(gson.toJson(res)), response.getOutputStream());
	}

	@RequestMapping("/ui/wf_metaworkflow.img")
	public void showMetaWorkflow(final HttpServletResponse response, @RequestParam(value = "id", required = true) final String id) throws Exception {

		String xml = isLookupClient.getSarasvatiMetaWorkflow(id);

		BufferedImage image = processGraphGenerator.getMetaWfDescImage(id, xml);
		sendImage(response, image);
	}

	@RequestMapping("/ui/wf_metaworkflow.edit")
	public void scheduleMetaWorkflow(final HttpServletResponse response, @RequestParam(value = "json", required = true) final String json) throws Exception {

		final Gson gson = new Gson();
		final AdvancedMetaWorkflowDescriptor info = gson.fromJson(json, AdvancedMetaWorkflowDescriptor.class);

		log.info("Updating workflow " + info.getName());

		final String xml = isLookupClient.getProfile(info.getWfId());
		boolean res = isRegistryClient.updateSarasvatiMetaWorkflow(info.getWfId(), xml, info);

		IOUtils.copy(new StringReader(gson.toJson(res)), response.getOutputStream());
	}

	@RequestMapping("/ui/wf_metaworkflow.clone")
	public void cloneMetaWf(final HttpServletResponse response,
			@RequestParam(value = "id", required = true) final String id,
			@RequestParam(value = "name", required = true) final String name) throws Exception {

		if (name.trim().length() > 0) {
			final Gson gson = new Gson();
			final String xml = isLookupClient.getProfile(id);
			SAXReader reader = new SAXReader();
			Document doc = reader.read(new StringReader(xml));
			doc.selectSingleNode("//METAWORKFLOW_NAME").setText(name);
			for (Object o : doc.selectNodes("//WORKFLOW")) {
				Element n = (Element) o;
				String atomWfXml = isLookupClient.getProfile(n.valueOf("@id"));
				String newAtomWfId = isRegistryClient.registerProfile(atomWfXml);
				n.addAttribute("id", newAtomWfId);
			}
			String res = isRegistryClient.registerProfile(doc.asXML());
			IOUtils.copy(new StringReader(gson.toJson(res)), response.getOutputStream());
		} else throw new IllegalArgumentException("Name is empty");
	}

	@RequestMapping("/ui/recentWfs.list")
	public void recentWfs(final HttpServletResponse response) throws IOException {
		final List<ProcessListEntry> res = Lists.newArrayList();
		for (String pid : graphProcessRegistry.listIdentifiers()) {
			res.add(new ProcessListEntry(pid, graphProcessRegistry.findProcess(pid)));
		}
		IOUtils.copy(new StringReader((new Gson()).toJson(res)), response.getOutputStream());
	}

	@RequestMapping("/ui/recentWfs.clear")
	public void clearRecentWfs(final HttpServletResponse response, @RequestParam(value = "json", required = true) final String json) throws IOException {
		for (Object id : (new Gson()).fromJson(json, List.class)) {
			graphProcessRegistry.unregisterProcess(id.toString());
		}
		recentWfs(response);
	}

	@RequestMapping("/ui/wf_proc.json")
	public void getProcessWorkflow(final HttpServletResponse response, @RequestParam(value = "id", required = true) final String id) throws Exception {
		final GraphProcess process = graphProcessRegistry.findProcess(id);

		final String mapContent = (process.getState() == ProcessState.Created) ? "" : processGraphGenerator.getProcessImageMap(id);

		String status = "";
		if (!process.isComplete()) {
			status = process.getState().toString().toUpperCase();
		} else if ("true".equals(process.getEnv().getAttribute(WorkflowsConstants.SYSTEM_COMPLETED_SUCCESSFULLY))) {
			status = "SUCCESS";
		} else {
			status = "FAILURE";
		}

		final String img = (process.getState() == ProcessState.Created) ? "file.get?src=notStarted.gif" : "wf_proc.img?id=" + id + "&t=" + DateUtils.now();
		final int priority = NumberUtils.toInt(process.getEnv().getAttribute(WorkflowsConstants.SYSTEM_WF_PRIORITY), WorkflowsConstants.DEFAULT_WF_PRIORITY);
		final String name = process.getGraph().getName();

		final AtomicWorkflowDescriptor wf = new AtomicWorkflowDescriptor(id, name, priority, status, mapContent, img, true, "auto", "RUNNING");

		IOUtils.copy(new StringReader(new Gson().toJson(wf)), response.getOutputStream());
	}

	@RequestMapping("/ui/wf_proc_node.json")
	public void getProcessWorkflowNode(final HttpServletResponse response,
			@RequestParam(value = "id", required = true) final String pid,
			@RequestParam(value = "node", required = true) final long nid) throws Exception {

		final NodeToken token = findNodeToken(pid, nid);

		final NodeTokenInfo info = (token == null) ? new NodeTokenInfo(findNodeName(pid, nid)) : new NodeTokenInfo(token);

		IOUtils.copy(new StringReader(new Gson().toJson(info)), response.getOutputStream());
	}

	private NodeToken findNodeToken(final String pid, final long nid) {
		final GraphProcess process = graphProcessRegistry.findProcess(pid);
		if (process != null) {
			for (NodeToken token : process.getNodeTokens()) {
				if (token.getNode().getId() == nid) return token;
			}
		}
		return null;
	}

	private String findNodeName(final String pid, final long nid) {
		final GraphProcess process = graphProcessRegistry.findProcess(pid);
		if (process != null) {
			for (Node node : process.getGraph().getNodes()) {
				if (node.getId() == nid) return node.getName();
			}
		}
		return "-";
	}

	@RequestMapping("/ui/wf_proc.img")
	public void showProcessWorkflow(final HttpServletResponse response, @RequestParam(value = "id", required = true) final String id) throws Exception {
		BufferedImage image = processGraphGenerator.getProcessImage(id);
		sendImage(response, image);
	}

	@RequestMapping("/ui/wf_proc.kill")
	public void killProcessWorkflow(final HttpServletResponse response, @RequestParam(value = "id", required = true) final String id) throws Exception {
		GraphProcess proc = graphProcessRegistry.findProcess(id);
		proc.setState(ProcessState.Canceled);
		getProcessWorkflow(response, id);
	}

	@RequestMapping("/ui/wf_journal.list")
	public void listWfJournal(final HttpServletResponse response) throws Exception {
		final Iterable<Map<String, String>> journal = Iterables.transform(dnetLogger, new JournalEntryFunction());
		IOUtils.copy(new StringReader(new Gson().toJson(Lists.newArrayList(journal))), response.getOutputStream());
	}

	@RequestMapping("/ui/wf_journal.range")
	public void rangeWfJournal(final HttpServletResponse response,
			@RequestParam(value = "start", required = true) final String start,
			@RequestParam(value = "end", required = true) final String end) throws Exception {
		DateTimeFormatter formatter = DateTimeFormat.forPattern("yyyy-MM-dd");
		DateTime startDate = formatter.parseDateTime(start);
		// System.out.println(startDate.toString());
		DateTime endDate = formatter.parseDateTime(end).plusHours(23).plusMinutes(59).plusSeconds(59);
		// System.out.println(endDate.toString());
		final Iterator<Map<String, String>> iter = dnetLogger.range(startDate.toDate(), endDate.toDate());
		final List<Map<String, String>> list = Lists.newArrayList(Iterators.transform(iter, new JournalEntryFunction()));
		IOUtils.copy(new StringReader(new Gson().toJson(list)), response.getOutputStream());
	}

	@RequestMapping("/ui/wf_journal.find")
	public void findWfJournalLogs(final HttpServletResponse response, @RequestParam(value = "id", required = true) final String wfId) throws Exception {
		final Iterator<Map<String, String>> iter = dnetLogger.find("system:profileId", wfId);
		final List<Map<String, String>> list = Lists.newArrayList(Iterators.transform(iter, new JournalEntryFunction()));
		IOUtils.copy(new StringReader(new Gson().toJson(list)), response.getOutputStream());
	}

	@RequestMapping("/ui/wf_journal.get")
	public void getWfJournalLog(final HttpServletResponse response, @RequestParam(value = "id", required = true) final String id) throws Exception {
		final Map<String, String> map = dnetLogger.findOne("system:processId", id);

		final List<String> keys = Lists.newArrayList(map.keySet());
		Collections.sort(keys);

		final List<Map<String, String>> list = Lists.transform(keys, new Function<String, Map<String, String>>() {

			@Override
			public Map<String, String> apply(final String k) {
				final Map<String, String> res = Maps.newHashMap();
				res.put("k", k);
				res.put("v", map.get(k));
				return res;
			}
		});

		IOUtils.copy(new StringReader(new Gson().toJson(list)), response.getOutputStream());
	}

	@RequestMapping("/ui/wf_atomic_workflow.enable")
	public void enableAtomicWf(final HttpServletResponse response,
			@RequestParam(value = "id", required = true) final String id,
			@RequestParam(value = "start", required = true) final String value) throws Exception {

		isRegistryClient.configureWorkflowStart(id, value);

		IOUtils.copy(new StringReader(new Gson().toJson(value)), response.getOutputStream());
	}

	@RequestMapping("/ui/wf_atomic_workflow.priority")
	public void changeAtomicWfPriority(final HttpServletResponse response,
			@RequestParam(value = "id", required = true) final String id,
			@RequestParam(value = "value", required = true) final int value) throws Exception {
		isRegistryClient.changeAtomicWfPriority(id, value);

		IOUtils.copy(new StringReader(new Gson().toJson(value)), response.getOutputStream());
	}

}
