package eu.dnetlib.functionality.modular.ui.workflows.util;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Resource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.tools.ServiceLocator;
import eu.dnetlib.functionality.modular.ui.workflows.objects.AdvancedMetaWorkflowDescriptor;
import eu.dnetlib.functionality.modular.ui.workflows.objects.AtomicWorkflowDescriptor;
import eu.dnetlib.functionality.modular.ui.workflows.objects.DatasourceWorkflowDescriptor;
import eu.dnetlib.functionality.modular.ui.workflows.objects.MetaWorkflowDescriptor;
import eu.dnetlib.functionality.modular.ui.workflows.objects.NodeInfo;
import eu.dnetlib.msro.workflows.util.WorkflowParam;

public class ISLookupClient {

	@Resource(name = "lookupLocator")
	private ServiceLocator<ISLookUpService> lookupLocator;

	private static final Log log = LogFactory.getLog(ISLookupClient.class);

	public List<String> listSimpleWorflowSections() {
		final String xquery = "distinct-values(//METAWORKFLOW_SECTION/text())";
		try {
			return lookupLocator.getService().quickSearchProfile(xquery);
		} catch (ISLookUpException e) {
			log.error("Error obtaining worflowSections", e);
			return Lists.newArrayList();
		}
	}

	public List<MetaWorkflowDescriptor> listMetaWorflowsForSection(final String name) {
		String query = queryForMetaWorkflowDescriptor(".//RESOURCE_TYPE/@value='MetaWorkflowDSResourceType' and .//METAWORKFLOW_SECTION='" + name + "'");

		try {
			return Lists.transform(lookupLocator.getService().quickSearchProfile(query), TransformerUtils.toMetaWorkflow);
		} catch (ISLookUpException e) {
			return Lists.newArrayList();
		}
	}

	public AtomicWorkflowDescriptor getAtomicWorkflow(final String id) {
		String s = getProfile(id);
		return TransformerUtils.toAtomicWorkflow.apply(s);
	}

	public List<DatasourceWorkflowDescriptor> listDatasources() {
		String query = queryForDatasourceWorkflowDescriptor("collection('/db/DRIVER/RepositoryServiceResources/RepositoryServiceResourceType')");

		try {
			return Lists.transform(lookupLocator.getService().quickSearchProfile(query), TransformerUtils.toDatasourceWorkflow);
		} catch (ISLookUpException e) {
			return Lists.newArrayList();
		}
	}

	public DatasourceWorkflowDescriptor getDatasourceWorkflow(final String id) {

		String query = queryForDatasourceWorkflowDescriptor("/*[.//RESOURCE_IDENTIFIER/@value='" + id + "']");

		try {
			String s = lookupLocator.getService().getResourceProfileByQuery(query);
			return TransformerUtils.toDatasourceWorkflow.apply(s);
		} catch (ISLookUpException e) {
			return null;
		}
	}

	private String queryForDatasourceWorkflowDescriptor(final String coll) {
		return "for $x in " + coll + " return <record>" + "<dsId>{$x//RESOURCE_IDENTIFIER/@value/string()}</dsId>"
				+ "<dsName>{$x//OFFICIAL_NAME/text()}</dsName>" + "<protocols>{$x//ACCESS_PROTOCOL}</protocols>"
				+ "<dsType>{$x//DATASOURCE_TYPE/text()}</dsType>" + "<metaworkflows>" + "{"
				+ "for $y in collection('/db/DRIVER/MetaWorkflowDSResources/MetaWorkflowDSResourceType') "
				+ "where $y//DATAPROVIDER/@id=$x//RESOURCE_IDENTIFIER/@value " + "return <workflow " + "id='{$y//RESOURCE_IDENTIFIER/@value/string()}' "
				+ "name='{$y//METAWORKFLOW_NAME/text()}' " + "family='{$y//METAWORKFLOW_NAME/@family/string()}' "
				+ "status='{$y//CONFIGURATION/@status/string()}' /> " + "}" + "</metaworkflows>" + "</record>";
	}

	private String queryForMetaWorkflowDescriptor(final String cond) {
		return "for $x in /*[" + cond + "]\n" + "return concat(" + "$x//RESOURCE_IDENTIFIER/@value, ' @@@ ', " + "$x//METAWORKFLOW_NAME, ' @@@ ', "
				+ "$x//METAWORKFLOW_NAME/@family, ' @@@ ', " + "$x//CONFIGURATION/@status)";
	}

	public String getProfile(final String id) {
		try {
			return lookupLocator.getService().getResourceProfile(id);
		} catch (ISLookUpException e) {
			log.error("Error finding profile: " + id, e);
			return null;
		}
	}

	public String getSarasvatiMetaWorkflow(final String id) {
		String s = getProfile(id);
		return TransformerUtils.toSarasvatiMetaWorkflow.evaluate(s);
	}

	public NodeInfo getNodeInfo(final String wfId, final String nodeName) {
		try {
			final String query = "/*[.//RESOURCE_IDENTIFIER/@value='" + wfId + "']//NODE[@name='" + nodeName + "']";
			final String xml = lookupLocator.getService().getResourceProfileByQuery(query);
			return TransformerUtils.toNodeInfo.apply(xml);
		} catch (Exception e) {
			log.error("Error accessing node " + nodeName + " of wf " + wfId, e);
			return new NodeInfo("", "", new ArrayList<WorkflowParam>(), new ArrayList<WorkflowParam>());
		}
	}

	public Set<String> getNotConfiguredNodes(final String id) {
		final String query = "for $x in (/*[.//RESOURCE_IDENTIFIER/@value='" + id + "']//NODE) "
				+ "where count($x//PARAM[@required='true' and string-length(normalize-space(.)) = 0]) > 0 " + "return $x/@name/string()";

		try {
			final List<String> list = lookupLocator.getService().quickSearchProfile(query);
			return Sets.newHashSet(list);
		} catch (Exception e) {
			log.error("Error executing xquery: " + query, e);
			return Sets.newHashSet();
		}
	}

	public AdvancedMetaWorkflowDescriptor getMetaWorkflow(final String id) {
		String s = getProfile(id);
		return TransformerUtils.toAdvancedMetaWorkflow.apply(s);
	}

	public Map<String, String> listRepoHiWorkflows() {
		final Map<String, String> map = Maps.newHashMap();

		final String query = "for $x in collection('/db/DRIVER/WorkflowDSResources/WorkflowDSResourceType') " + "where $x//WORKFLOW_TYPE='REPO_HI' "
				+ "return concat($x//RESOURCE_IDENTIFIER/@value, ' @@@ ', $x//WORKFLOW_NAME)";

		try {
			for (String s : lookupLocator.getService().quickSearchProfile(query)) {
				String[] arr = s.split("@@@");
				map.put(arr[0].trim(), arr[1].trim());
			}
		} catch (Exception e) {
			log.error("Error executing xquery: " + query, e);
		}
		return map;
	}

	public List<String> listMetaWorflowsForWfId(final String wfId) {
		final String query = "for $x in collection('/db/DRIVER/MetaWorkflowDSResources/MetaWorkflowDSResourceType') where $x//WORKFLOW/@id='" + wfId
				+ "' return $x//RESOURCE_IDENTIFIER/@value/string()";
		try {
			return lookupLocator.getService().quickSearchProfile(query);
		} catch (ISLookUpException e) {
			log.error("Error executing xquery: " + query, e);
			return Lists.newArrayList();
		}
	}

	public boolean isExecutable(final String metaWfId) {
		final String query = "for $x in collection('/db/DRIVER/MetaWorkflowDSResources/MetaWorkflowDSResourceType')[.//RESOURCE_IDENTIFIER/@value = '"
				+ metaWfId
				+ "']//WORKFLOW/@id \n"
				+ "for $y in collection('/db/DRIVER/WorkflowDSResources/WorkflowDSResourceType')[.//RESOURCE_IDENTIFIER/@value = $x]//PARAM[@required='true' and string-length(normalize-space(.)) = 0] \n"
				+ "return $y";
		try {
			return lookupLocator.getService().quickSearchProfile(query).isEmpty();
		} catch (ISLookUpException e) {
			log.error("Error executing xquery: " + query, e);
			return false;
		}
	}

	public List<String> listWfFamilies() {
		final String query = "distinct-values(for $x in collection('/db/DRIVER/MetaWorkflowDSResources/MetaWorkflowDSResourceType') "
				+ "where string-length($x//DATAPROVIDER/@id) > 0 return $x//METAWORKFLOW_NAME/@family/string())";
		try {
			return lookupLocator.getService().quickSearchProfile(query);
		} catch (ISLookUpException e) {
			log.error("Error executing xquery: " + query, e);
			return Lists.newArrayList();
		}
	}

}
