package com.finconsgroup.itserr.marketplace.zenodo.bs.messaging.impl;

import com.finconsgroup.itserr.marketplace.zenodo.bs.messaging.EventProducer;
import com.finconsgroup.itserr.marketplace.zenodo.bs.messaging.configuration.properties.MessagingConfigurationProperties;
import com.finconsgroup.itserr.marketplace.zenodo.bs.messaging.dto.ExportRequestCreatedMessageBodyDto;
import com.finconsgroup.itserr.messaging.exception.WP2MessagingException;
import com.finconsgroup.itserr.messaging.producer.MessageProducer;
import com.finconsgroup.itserr.messaging.producer.ProducerRegistry;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

import java.util.Map;

import static com.finconsgroup.itserr.marketplace.zenodo.bs.util.Constants.SERVICE_FILTER_KEY;
import static com.finconsgroup.itserr.marketplace.zenodo.bs.util.Constants.SERVICE_FILTER_VALUE;

/**
 * Default implementation of {@link EventProducer} that publishes resource events (create, update, delete) for news.
 */
@Component
@ConditionalOnProperty(name = "messaging.enabled", havingValue = "true")
@Slf4j
public class DefaultEventProducer implements EventProducer {

    /** Message producer used to publish events to the messaging system */
    private final MessageProducer messageProducer;

    /** Configuration properties for messaging */
    private final MessagingConfigurationProperties messagingProperties;

    /**
     * Creates a new DefaultEventProducer instance.
     *
     * @param messagingProperties Configuration properties for messaging
     * @param producerRegistry Registry to get message producer from
     */
    public DefaultEventProducer(
            MessagingConfigurationProperties messagingProperties,
            ProducerRegistry producerRegistry
    ) {
        this.messagingProperties = messagingProperties;
        this.messageProducer = producerRegistry.getMessageProducer(this.messagingProperties.getEventProducerName());
    }

    /**
     * Publishes an event indicating that a new export request has been created.
     *
     * @param resource The DTO containing the export request details that was created
     */
    @Override
    public void publishExportRequestCreatedEvent(@NonNull final ExportRequestCreatedMessageBodyDto resource) {
        publish(resource.getId(), resource, messagingProperties.getExportRequestCreatedEventType());
    }

    /**
     * Sends a message to the messaging system.
     *
     * @param resourceId the resource id
     * @param resource the payload of the message
     * @param eventType the type of event to publish
     * @throws WP2MessagingException if message publishing fails
     */
    private <T> void publish(
            @NonNull String resourceId,
            @NonNull T resource,
            @NonNull String eventType) {
        try {
            Map<String, T> messageMap = Map.of(resourceId, resource);
            Map<String, String> filterProperties = Map.of(SERVICE_FILTER_KEY, SERVICE_FILTER_VALUE);
            log.debug("Sending event message, messageMap:{}, eventType: {}, source: {}, filterProperties: {}",
                    messageMap, eventType, messagingProperties.getSource(), filterProperties);
            messageProducer.publishEvent(messageMap, eventType, messagingProperties.getSource(), filterProperties);
            log.info("Successfully published event message for resource with id: {}", resourceId);
        } catch (Exception e) {
            String errorMessage = "Error occurred while sending event message: %s".formatted(e.getMessage());
            throw new WP2MessagingException(errorMessage, e);
        }

    }

}
