/**
 * Global exception handler for REST controllers in the zenodo system. This class handles custom exceptions related to zenodo and category operations and
 * returns appropriate HTTP error responses.
 *
 * <p>Example usage:
 * <pre>
 * When a ZenodoNotFoundException is thrown, this handler returns a 404 NOT FOUND response.
 * </pre>
 * </p>
 */
package com.finconsgroup.itserr.marketplace.zenodo.bs.exception;

import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.context.request.WebRequest;

import static com.finconsgroup.itserr.marketplace.core.web.exception.ExceptionUtils.createResponseEntityError;
import static com.finconsgroup.itserr.marketplace.core.web.exception.ExceptionUtils.logError;

/**
 * Global exception handler for REST controllers.
 */
@RestControllerAdvice
public class CustomExceptionHandler {

    /**
     * Handles exceptions when source item ID is missing from the request.
     *
     * @param ex The NoSourceItemIdException that was thrown
     * @param request The web request that caused the exception
     * @return ResponseEntity containing error details with BAD_REQUEST status
     */
    @ExceptionHandler(NoSourceItemIdException.class)
    private ResponseEntity<ErrorResponseDto> handleNoSourceItemIdException(
            final NoSourceItemIdException ex,
            final WebRequest request) {
        logError(ex, request);
        return createResponseEntityError(HttpStatus.BAD_REQUEST, ex.getMessage());
    }

    /**
     * Handles exceptions when Zenodo authentication code is missing.
     *
     * @param ex The NoZenodoAuthCodeException that was thrown
     * @param request The web request that caused the exception
     * @return ResponseEntity containing error details with BAD_REQUEST status
     */
    @ExceptionHandler(NoZenodoAuthCodeException.class)
    private ResponseEntity<ErrorResponseDto> handleNoZenodoAuthCodeException(
            final NoZenodoAuthCodeException ex,
            final WebRequest request) {
        logError(ex, request);
        return createResponseEntityError(HttpStatus.BAD_REQUEST, ex.getMessage());
    }

    /**
     * Handles exceptions when the requested source item cannot be found.
     *
     * @param ex The SourceItemNotFoundException that was thrown
     * @param request The web request that caused the exception
     * @return ResponseEntity containing error details with NOT_FOUND status
     */
    @ExceptionHandler(SourceItemNotFoundException.class)
    private ResponseEntity<ErrorResponseDto> handleSourceItemNotFoundException(
            final SourceItemNotFoundException ex,
            final WebRequest request) {
        logError(ex, request);
        return createResponseEntityError(HttpStatus.NOT_FOUND, ex.getMessage());
    }

    /**
     * Handles exceptions that occur during source item retrieval.
     *
     * @param ex The SourceItemRetrieveException that was thrown
     * @param request The web request that caused the exception
     * @return ResponseEntity containing error details with INTERNAL_SERVER_ERROR status
     */
    @ExceptionHandler(SourceItemRetrieveException.class)
    private ResponseEntity<ErrorResponseDto> handleSourceItemRetrieveException(
            final SourceItemRetrieveException ex,
            final WebRequest request) {
        logError(ex, request);
        return createResponseEntityError(HttpStatus.INTERNAL_SERVER_ERROR, ex.getMessage());
    }

    /**
     * Handles exceptions when access to a source item is not allowed.
     *
     * @param ex The UnallowedSourceItemException that was thrown
     * @param request The web request that caused the exception
     * @return ResponseEntity containing error details with FORBIDDEN status
     */
    @ExceptionHandler(UnallowedSourceItemException.class)
    private ResponseEntity<ErrorResponseDto> handleUnallowedSourceItemException(
            final UnallowedSourceItemException ex,
            final WebRequest request) {
        logError(ex, request);
        return createResponseEntityError(HttpStatus.FORBIDDEN, ex.getMessage());
    }

    /**
     * Handles exceptions when Zenodo authentication fails.
     *
     * @param ex The ZenodoAuthFailureException that was thrown
     * @param request The web request that caused the exception
     * @return ResponseEntity containing error details with UNPROCESSABLE_ENTITY status
     */
    @ExceptionHandler(ZenodoAuthFailureException.class)
    private ResponseEntity<ErrorResponseDto> handleZenodoAuthFailureException(
            final ZenodoAuthFailureException ex,
            final WebRequest request) {
        logError(ex, request);
        return createResponseEntityError(HttpStatus.UNPROCESSABLE_ENTITY, ex.getMessage());
    }

}


