package com.finconsgroup.itserr.marketplace.userprofile.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.FetchType;
import jakarta.persistence.JoinColumn;
import jakarta.persistence.ManyToOne;
import jakarta.persistence.Table;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.experimental.SuperBuilder;

import java.time.Instant;
import java.util.UUID;

/**
 * Entity class representing an archived endorsement of an expertise of a user profile.
 */
@Entity
@Table(name = "archived_endorsement")
@SuperBuilder
@Getter
@Setter
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class ArchivedEndorsementEntity extends AbstractUUIDEntity {

    @ManyToOne(fetch = FetchType.LAZY)
    @JoinColumn(name = "expertise_id", nullable = false)
    private ArchivedExpertiseEntity expertise;

    /**
     * Unique identifier of the user providing the endorsement.
     */
    @Column(name = "endorser_id")
    private UUID endorserId;

    /**
     * The ordering column.
     */
    @Column(name = "endorsement_order", nullable = false)
    private long endorsementOrder;

    /**
     * The timestamp when the event was created.
     */
    @Column(name = "creation_time", nullable = false)
    private Instant creationTime;

    /**
     * The timestamp when the event was last updated.
     */
    @Column(name = "update_time", nullable = false)
    private Instant updateTime;

    /**
     * The timestamp when the event was last archived.
     */
    @Column(name = "archive_time", nullable = false)
    private Instant archiveTime;

    /**
     * The version field used for optimistic locking.
     * <p>
     * This value is automatically managed by JPA to detect concurrent updates.
     * Each time the entity is updated, the version is incremented.
     * If two transactions try to update the same entity simultaneously,
     * JPA will detect the conflict based on this version
     * and throw an {@link jakarta.persistence.OptimisticLockException}.
     */
    @Column(name = "version", nullable = false)
    private long version;

    @Override
    public void prePersist() {
        super.prePersist();
        Instant now = Instant.now();
        if (creationTime == null) {
            creationTime = now;
        }
        if (updateTime == null) {
            updateTime = now;
        }
        if (archiveTime == null) {
            archiveTime = now;
        }
    }

}
