package com.finconsgroup.itserr.marketplace.userprofile.dm.controller;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.userprofile.dm.api.UserProfileApi;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.AuthenticatedUser;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputAddProjectToUserProfilesDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputFindUserProfilesByIdsDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputFindUserProfilesByInterestsDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputFindUserProfilesByPrincipalsDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputFindUserProfilesByTokenInfoDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputPatchUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputPatchUserProfileProjectDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputRemoveProjectFromUserProfilesDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputUpdateUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.OutputPatchUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.OutputUserProfileAutoCompleteDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.OutputUserProfileFolderDetailsDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.service.UserProfileService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort.Direction;
import org.springframework.lang.NonNull;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.UUID;

/**
 * REST controller class for handling UserProfile related API requests.
 *
 * <p>Implements the {@link UserProfileApi} interface.</p>
 */
@Slf4j
@RequiredArgsConstructor
@RestController
public class UserProfileController implements UserProfileApi {

    private final UserProfileService userProfileService;

    @Override
    public OutputUserProfileDto create(final InputUserProfileDto inputUserProfileDto) {
        return this.userProfileService.create(AuthenticatedUser.fromJwt(), inputUserProfileDto);
    }

    @Override
    public Page<OutputUserProfileDto> findAll(
            final int pageNumber,
            final int pageSize,
            final String sort,
            final Direction direction
    ) {
        Pageable pageable = PageRequest.of(pageNumber, pageSize, direction, sort);
        return this.userProfileService.findAll(pageable);
    }

    @Override
    public OutputUserProfileDto findById() {
        return userProfileService.findById(AuthenticatedUser.fromJwt());
    }

    @Override
    public OutputUserProfileDto getById(UUID profileId) {
        return userProfileService.getById(profileId);
    }

    @Override
    public Page<OutputUserProfileDto> findAllByIds(
            InputFindUserProfilesByIdsDto dto,
            final int pageNumber,
            final int pageSize,
            final String sort,
            final Direction direction) {
        Pageable pageable = PageRequest.of(pageNumber, pageSize, direction, sort);
        return userProfileService.findAllByIds(dto, pageable);
    }

    @Override
    public Page<OutputUserProfileAutoCompleteDto> getAutoCompletions(
            String terms,
            final int pageNumber,
            final int pageSize,
            final String sort,
            final Direction direction) {
        Pageable pageable = PageRequest.of(pageNumber, pageSize, direction, sort);
        return userProfileService.getAutoCompletions(terms, pageable);
    }

    @Override
    public OutputUserProfileDto updateById(
            final InputUpdateUserProfileDto inputUpdateUserProfileDto
    ) {
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        return this.userProfileService.updateById(userId, inputUpdateUserProfileDto);
    }

    @Override
    public OutputUserProfileFolderDetailsDto getUserProfileFolderId() {
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        return this.userProfileService.getUserProfileFolderId(userId);
    }

    @Override
    public OutputPatchUserProfileDto patchUserProfile(InputPatchUserProfileDto patchUserProfileDto) {
        return this.userProfileService.patchUserProfileInfo(patchUserProfileDto);
    }

    @Override
    public Page<OutputUserProfileDto> findAllByPrincipals(InputFindUserProfilesByPrincipalsDto inputFindUserProfilesByPrincipalsDto, int pageNumber, int pageSize, String sort, Direction direction) {
        Pageable pageable = PageRequest.of(pageNumber, pageSize, direction, sort);
        return userProfileService.findAllByPrincipals(inputFindUserProfilesByPrincipalsDto, pageable);
    }

    @Override
    public List<OutputUserProfileDto> addProjectToUserProfiles(@NonNull InputAddProjectToUserProfilesDto inputAddProjectToUserProfilesDto) {
        return userProfileService.addProjectToUserProfiles(inputAddProjectToUserProfilesDto);
    }

    @Override
    public List<OutputUserProfileDto> removeProjectFromUserProfiles(@NonNull InputRemoveProjectFromUserProfilesDto inputRemoveProjectFromUserProfilesDto) {
        return userProfileService.removeProjectFromUserProfiles(inputRemoveProjectFromUserProfilesDto);
    }

    @Override
    public List<OutputUserProfileDto> patchUserProfileProject(InputPatchUserProfileProjectDto inputPatchUserProfileProjectDto) {
        return userProfileService.patchUserProfileProject(inputPatchUserProfileProjectDto);
    }

    @Override
    public List<OutputUserProfileDto> findAllByTokenInfo(InputFindUserProfilesByTokenInfoDto inputFindUserProfilesByTokenInfoDto) {
        return userProfileService.findAllByTokenInfo(inputFindUserProfilesByTokenInfoDto);
    }

    @Override
    public void activate() {
        userProfileService.processUserProfileStatusChange(JwtTokenHolder.getUserIdOrThrow(), true);
    }

    @Override
    public void deactivate() {
        userProfileService.processUserProfileStatusChange(JwtTokenHolder.getUserIdOrThrow(), false);
    }

    @Override
    public List<UUID> findMatchingInterests(InputFindUserProfilesByInterestsDto dto) {
        return userProfileService.findMatchingInterests(dto.getStringsToCheck());
    }
}
