package com.finconsgroup.itserr.marketplace.userprofile.dm.api;

import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputAddProjectToUserProfilesDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputFindUserProfilesByIdsDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputFindUserProfilesByInterestsDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputFindUserProfilesByPrincipalsDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputFindUserProfilesByTokenInfoDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputPatchUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputPatchUserProfileProjectDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputRemoveProjectFromUserProfilesDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputUpdateUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.OutputPatchUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.OutputUserProfileAutoCompleteDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.OutputUserProfileFolderDetailsDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.validation.annotation.ValidInputCreateOrUpdateUserProfileDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.NotNull;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import jakarta.validation.constraints.Size;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Sort;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;

import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.List;
import java.util.UUID;


/**
 * This interface defines the contract for REST API endpoints related to UserProfile Management.
 *
 * <p>
 * It provides endpoints for retrieving, creating, updating, and deleting UserProfiles.
 * </p>
 *
 * <p>Example usage:
 * <pre>
 * POST   /userprofile - Create a new userprofile.
 * GET    /userprofile - Retrieve a paginated list of all UserProfiles.
 * GET    /userprofile/{userprofileId} - Retrieve a userprofile by id.
 * PUT    /userprofile/{userprofileId} - Update an existing userprofile.
 * DELETE /userprofile/{userprofileId} - Delete a userprofile.
 * </pre>
 * </p>
 */
@Tag(
        name = "UserProfile",
        description = "The UserProfile API: it provides endpoints for " +
                "retrieving, creating, updating, and deleting UserProfiles."
)
@SecurityRequirement(name = "BearerAuth")
public interface UserProfileApi {

    /**
     * Retrieves a paginated list of all UserProfiles.
     *
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of UserProfiles per page (default is 10)
     * @param sort       the field to sort by (default is "id")
     * @param direction  the direction of sorting (default is ascending)
     * @return a page of {@link OutputUserProfileDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "find all UserProfiles",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @GetMapping(value = "/profiles", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    Page<OutputUserProfileDto> findAll(
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "id", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction
    );

    /**
     * Retrieves an userprofile by id using JWT token.
     *
     * @return the found {@link OutputUserProfileDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "find userprofile by id",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found",
                            content = {
                                    @Content(
                                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                                            schema = @Schema(implementation = ErrorResponseDto.class)
                                    )
                            }
                    ),
            }
    )
    @GetMapping(value = "/profile", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputUserProfileDto findById();

    /**
     * Retrieves an userprofile by profileId.
     *
     * @param profileId of the userprofile to fetch
     * @return the found {@link OutputUserProfileDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "get userprofile by profileId",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found",
                            content = {
                                    @Content(
                                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                                            schema = @Schema(implementation = ErrorResponseDto.class)
                                    )
                            }
                    ),
            }
    )
    @GetMapping(value = "/profiles/{profileId}", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputUserProfileDto getById(@PathVariable("profileId") UUID profileId);

    /**
     * Retrieves a paginated list of all user profiles for provided ids.
     *
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of UserProfiles per page (default is 10)
     * @param sort       the field to sort by (default is "id")
     * @param direction  the direction of sorting (default is ascending)
     * @return the list of found {@link OutputUserProfileDto}s and HTTP status 200 (OK)
     */
    @Operation(
            summary = "find user profiles by ids",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK")
            }
    )
    @PostMapping(value = "/profiles-by-ids", consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    Page<OutputUserProfileDto> findAllByIds(
            @Valid @RequestBody InputFindUserProfilesByIdsDto dto,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "id", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction);

    /**
     * Retrieves a limited list of autocompletion results matching the provided terms from user profile table.
     *
     * @param terms the terms to search for
     * @return the list of {@link OutputUserProfileAutoCompleteDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Perform user profile search for autocompletion and retrieve the results",
            description = "Perform user profile search for autocompletion based on the terms provided and retrieve results"
    )
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "OK")
    })
    @ResponseStatus(HttpStatus.OK)
    @GetMapping(value = "/profiles/auto-complete", produces = MediaType.APPLICATION_JSON_VALUE)
    Page<OutputUserProfileAutoCompleteDto> getAutoCompletions(
            @NotNull @Size(min = 3) @RequestParam("terms") String terms,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "id", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction);

    /**
     * Creates a new userprofile.
     *
     * @param inputUserProfileDto the input data transfer object containing userprofile details
     * @return the created {@link OutputUserProfileDto} and HTTP status 201 (Created)
     */
    @Operation(
            summary = "create userprofile for the authenticated user",
            responses = {
                    @ApiResponse(responseCode = "201", description = "Created"),
                    @ApiResponse(responseCode = "409", description = "Already Exists",
                            content = {
                                    @Content(
                                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                                            schema = @Schema(implementation = ErrorResponseDto.class)
                                    )
                            }
                    ),}
    )
    @PostMapping(
            value = "/profile",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.CREATED)
    OutputUserProfileDto create(@Valid @ValidInputCreateOrUpdateUserProfileDto @RequestBody InputUserProfileDto inputUserProfileDto);

    /**
     * Updates an existing userprofile by id.
     *
     * @param inputUpdateUserProfileDto the input data transfer object containing updated userprofile details
     * @return the updated {@link OutputUserProfileDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "update userprofile by id",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found",
                            content = {
                                    @Content(
                                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                                            schema = @Schema(implementation = ErrorResponseDto.class)
                                    )
                            }),
            }
    )
    @PutMapping(
            value = "/profile",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputUserProfileDto updateById(
            @Valid @ValidInputCreateOrUpdateUserProfileDto @RequestBody InputUpdateUserProfileDto inputUpdateUserProfileDto
    );

    /**
     * Retrieves the folderId of the user profile using JWT user token Id.
     *
     * @return the found {@link OutputUserProfileDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "find userprofile by id",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found",
                            content = {
                                    @Content(
                                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                                            schema = @Schema(implementation = ErrorResponseDto.class)
                                    )
                            }),
            }
    )
    @GetMapping(value = "/profile/folder", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputUserProfileFolderDetailsDto getUserProfileFolderId();

    /**
     * Patch the user info.
     *
     * @param patchUserProfileDto which contains information that needs to be patched for an existing entity
     * @return the found {@link OutputPatchUserProfileDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "patch user profile by id",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found",
                            content = {
                                    @Content(
                                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                                            schema = @Schema(implementation = ErrorResponseDto.class)
                                    )
                            }),
            }
    )
    @PatchMapping(value = "/profile", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputPatchUserProfileDto patchUserProfile(@RequestBody InputPatchUserProfileDto patchUserProfileDto);

    /**
     * Retrieves a paginated list of all user profiles for provided principals.
     *
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of UserProfiles per page (default is 10)
     * @param sort       the field to sort by (default is "id")
     * @param direction  the direction of sorting (default is ascending)
     * @return the list of found {@link OutputUserProfileDto}s and HTTP status 200 (OK)
     */
    @Operation(
            summary = "find user profiles by principals",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK")
            }
    )
    @PostMapping(value = "/profiles-by-principals", consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    Page<OutputUserProfileDto> findAllByPrincipals(
            @Valid @RequestBody InputFindUserProfilesByPrincipalsDto inputFindUserProfilesByPrincipalsDto,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "id", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction);

    /**
     * add project to existing userprofile for given ids.
     *
     * @param inputAddProjectToUserProfilesDto contains the list of userIds and object containing updated userprofile project details
     * @return the updated {@link List<OutputUserProfileDto>} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "add project to userprofile for given ids",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
            }
    )
    @PostMapping(
            value = "/profiles/project",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    List<OutputUserProfileDto> addProjectToUserProfiles(
            @Valid @RequestBody InputAddProjectToUserProfilesDto inputAddProjectToUserProfilesDto
    );

    /**
     * removes project from existing userprofile for given ids.
     *
     * @param inputRemoveProjectFromUserProfilesDto contains the list of userIds and object containing updated userprofile project details
     * @return the updated {@link List<OutputUserProfileDto>} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "removes project from userprofile for given ids",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
            }
    )
    @DeleteMapping(
            value = "/profiles/project",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    List<OutputUserProfileDto> removeProjectFromUserProfiles(
            @Valid @RequestBody InputRemoveProjectFromUserProfilesDto inputRemoveProjectFromUserProfilesDto
    );

    /**
     * update project for existing userprofile for given ids.
     *
     * @param inputPatchUserProfileProjectDto contains the list of userIds and patch updates for userprofile project details
     * @return the updated {@link List<OutputUserProfileDto>} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "update project for existing userprofile for given ids",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
            }
    )
    @PatchMapping(
            value = "/profiles/projects",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    List<OutputUserProfileDto> patchUserProfileProject(
            @Valid @RequestBody InputPatchUserProfileProjectDto inputPatchUserProfileProjectDto
    );

    /**
     * Retrieves a list of all user profiles for provided mixed token info.
     *
     * @return the list of found {@link OutputUserProfileDto}s and HTTP status 200 (OK)
     */
    @Operation(
            summary = "find user profiles by mixed info carried by the token",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK")
            }
    )
    @PostMapping(value = "/profiles-by-token-infos", consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    List<OutputUserProfileDto> findAllByTokenInfo(
            @Valid @RequestBody InputFindUserProfilesByTokenInfoDto inputFindUserProfilesByPrincipalsDto);

    /**
     * activate the user profile.
     */
    @Operation(
            summary = "activate a user profile by ID",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found",
                            content = {
                                    @Content(
                                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                                            schema = @Schema(implementation = ErrorResponseDto.class)
                                    )
                            }),
            },
            hidden = true
    )
    @PostMapping(value = "/profile/activate", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    void activate();

    /**
     * deactivate the user profile.
     */
    @Operation(
            summary = "deactivate a user profile by ID",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found",
                            content = {
                                    @Content(
                                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                                            schema = @Schema(implementation = ErrorResponseDto.class)
                                    )
                            }),
            },
            hidden = true
    )
    @DeleteMapping(value = "/profile/activate", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    void deactivate();

    /**
     * Retrieves the list of user profile ids whose interests match at least one the provided list of strings.
     *
     * @param dto the input to match interests against
     * @return the matching {@link List} of user profile ids
     */
    @Operation(
            summary = "find user profiles by matching interests",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK")
            }
    )
    @PostMapping(value = "/profile-ids-by-interests", consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    List<UUID> findMatchingInterests(@Valid @RequestBody InputFindUserProfilesByInterestsDto dto);
}
