package com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.helper;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputAddProjectToUserProfilesDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputPatchUserProfileProjectDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputRemoveProjectFromUserProfilesDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.InputProjectDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.dto.CatalogItemMessagingStatusAdditionalDataDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.dto.EventDataWrapper;
import com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.dto.InstitutionalPageStatusChangeNotificationData;
import com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.dto.JoinRequestStatusChangeNotificationData;
import com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.dto.MemberInvitationStatusChangeNotificationData;
import com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.dto.MembershipChangeNotificationData;
import com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.dto.MembershipRemovalNotificationData;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.UUID;

/**
 * Helper class for messaging-related data conversion operations.
 */
@Component
@RequiredArgsConstructor
@Slf4j
public class MessagingHelper {

    private final ObjectMapper objectMapper;

    public InstitutionalPageStatusChangeNotificationData convertToInstitutionalPageStatusChangeNotificationData(Object additionalData) {
        return convertData(additionalData, InstitutionalPageStatusChangeNotificationData.class,
                "Failed to convert additional data to InstitutionalPageStatusChangeNotificationData");
    }

    public JoinRequestStatusChangeNotificationData convertToJoinRequestStatusChangeNotificationData(Object additionalData) {
        return convertData(additionalData, JoinRequestStatusChangeNotificationData.class,
                "Failed to convert additional data to JoinRequestStatusChangeNotificationData");
    }

    public MemberInvitationStatusChangeNotificationData convertToMemberInvitationStatusChangeNotificationData(Object additionalData) {
        return convertData(additionalData, MemberInvitationStatusChangeNotificationData.class,
                "Failed to convert additional data to MemberInvitationStatusChangeNotificationData");
    }

    public MembershipChangeNotificationData convertToMembershipChangeNotificationData(Object additionalData) {
        return convertData(additionalData, MembershipChangeNotificationData.class,
                "Failed to convert additional data to MembershipChangeNotificationData");
    }

    public MembershipRemovalNotificationData convertToMembershipRemovalNotificationData(Object additionalData) {
        return convertData(additionalData, MembershipRemovalNotificationData.class,
                "Failed to convert additional data to MembershipRemovalNotificationData");
    }

    public CatalogItemMessagingStatusAdditionalDataDto convertToCatalogItemMessagingStatusAdditionalDataDto(Object additionalData) {
        return convertData(additionalData, CatalogItemMessagingStatusAdditionalDataDto.class,
                "Failed to convert additional data to CatalogItemMessagingStatusAdditionalDataDto");
    }

    private <T> T convertData(Object additionalData, Class<T> targetType, String errorMessage) {
        try {
            return objectMapper.convertValue(additionalData, targetType);
        } catch (IllegalArgumentException e) {
            log.error("{}: {}", errorMessage, additionalData, e);
            throw new IllegalArgumentException("Invalid additional data format", e);
        }
    }

    public InputAddProjectToUserProfilesDto buildInputAddProjectToUserProfilesDto(@NonNull UUID projectId, @NonNull String projectName, @NonNull UUID userId,
                                                                                  UUID rootInstitutionalPageId, String rootInstitutionalPageName,
                                                                                  Boolean wpLeader) {

        InputProjectDto inputProjectDto = buildProjectDto(projectId, projectName, rootInstitutionalPageId, rootInstitutionalPageName, wpLeader);
        return InputAddProjectToUserProfilesDto.builder()
                .userIds(List.of(userId))
                .project(inputProjectDto)
                .build();
    }

    public InputAddProjectToUserProfilesDto buildInputAddProjectToUserProfilesDto(@NonNull EventDataWrapper eventDataWrapper) {
        InputProjectDto inputProjectDto = buildProjectDto(eventDataWrapper);
        return InputAddProjectToUserProfilesDto.builder()
                .userIds(eventDataWrapper.getUserIds())
                .project(inputProjectDto)
                .build();
    }

    public InputRemoveProjectFromUserProfilesDto buildInputRemoveProjectFromUserProfilesDto(@NonNull UUID projectId, @NonNull List<UUID> userIds) {
        return InputRemoveProjectFromUserProfilesDto.builder()
                .userIds(userIds)
                .projectId(projectId)
                .build();
    }

    public InputRemoveProjectFromUserProfilesDto buildInputRemoveProjectFromUserProfilesDto(@NonNull EventDataWrapper eventDataWrapper) {
        return InputRemoveProjectFromUserProfilesDto.builder()
                .userIds(eventDataWrapper.getUserIds())
                .projectId(UUID.fromString(eventDataWrapper.getResourceId()))
                .build();
    }

    private InputProjectDto buildProjectDto(UUID projectId, String projectName, UUID rootInstitutionalPageId,
                                            String rootInstitutionalPageName, Boolean wpLeader) {
        InputProjectDto inputProjectDto = InputProjectDto.builder()
                .projectId(projectId)
                .displayName(projectName)
                .wpLead(wpLeader)
                .build();

        if (rootInstitutionalPageId != null) {
            inputProjectDto.setRootProjectId(rootInstitutionalPageId);
        }

        if (rootInstitutionalPageName != null) {
            inputProjectDto.setRootProjectDisplayName(rootInstitutionalPageName);
        }
        return inputProjectDto;
    }

    private InputProjectDto buildProjectDto(EventDataWrapper eventDataWrapper) {
        InputProjectDto inputProjectDto = InputProjectDto.builder()
                .projectId(UUID.fromString(eventDataWrapper.getResourceId()))
                .displayName(eventDataWrapper.getName())
                .wpLead(eventDataWrapper.getWpLeader())
                .build();

        if (eventDataWrapper.getRootInstitutionalPageId() != null) {
            inputProjectDto.setRootProjectId(eventDataWrapper.getRootInstitutionalPageId());
        }

        if (eventDataWrapper.getRootInstitutionalPageName() != null) {
            inputProjectDto.setRootProjectDisplayName(eventDataWrapper.getRootInstitutionalPageName());
        }
        return inputProjectDto;
    }

    public InputPatchUserProfileProjectDto buildInputPatchUserProfileProjectDto(@NonNull List<UUID> projectIds,
                                                                                @NonNull List<UUID> userIds,
                                                                                Boolean wpLeader) {
        return InputPatchUserProfileProjectDto.builder()
                .projectIdIds(projectIds)
                .userIds(userIds)
                .wpLead(wpLeader)
                .build();
    }
}