package com.finconsgroup.itserr.marketplace.search.dm.controller;

import com.finconsgroup.itserr.marketplace.search.dm.api.NewsApi;
import com.finconsgroup.itserr.marketplace.search.dm.dto.InputNewsDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputNewsDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputNewsLocalSearchDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchAutoCompleteDto;
import com.finconsgroup.itserr.marketplace.search.dm.service.NewsService;
import com.finconsgroup.itserr.marketplace.search.dm.util.SortUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.propertyeditors.StringTrimmerEditor;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.Map;

/**
 * Spring Boot REST Controller implementation for the {@link NewsApi} endpoints.
 */
@RestController
@RequiredArgsConstructor
@Slf4j
public class NewsController implements NewsApi {
    private final NewsService newsService;

    @InitBinder
    public void initBinder(WebDataBinder binder) {
        // trim strings before triggering validation, but do not convert to null
        // as otherwise it throws - MissingServletRequestParameterException
        binder.registerCustomEditor(String.class, new StringTrimmerEditor(false));
    }

    @NonNull
    @Override
    public OutputNewsDto upsertNews(String id, InputNewsDto dto) {
        return newsService.upsertDocument(dto);
    }

    @NonNull
    @Override
    public OutputNewsDto getNews(String id) {
        return newsService.getDocument(id);
    }

    @Override
    public void deleteNews(String id) {
        newsService.deleteDocument(id);
    }

    @Override
    public void deleteAll() {
        newsService.deleteAll();
    }

    @NonNull
    @Override
    public List<OutputGlobalSearchAutoCompleteDto> getAutoCompletions(@NonNull String terms) {
        return newsService.getAutoCompletions(terms);
    }

    @NonNull
    @Override
    public Page<OutputNewsLocalSearchDto> getLocalSearch(@Nullable String terms,
                                                          @Nullable String filters,
                                                          int pageNumber,
                                                          int pageSize,
                                                          String sort,
                                                          Sort.Direction direction,
                                                         String multiSort) {

        Sort sortToApply = SortUtils.buildSort(sort, direction, multiSort, ":", Map.of());
        Pageable pageable = PageRequest.of(pageNumber, pageSize, sortToApply);
        return newsService.getLocalSearch(terms, filters, pageable);
    }
}
