package com.finconsgroup.itserr.marketplace.search.bs.messaging;

import com.finconsgroup.itserr.marketplace.search.bs.client.dm.SearchDmClient;
import com.finconsgroup.itserr.marketplace.search.bs.config.properties.MessagingCatalogBsProperties;
import com.finconsgroup.itserr.marketplace.search.bs.config.properties.SearchBsConfigurationProperties;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.Objects;

/**
 * Concrete implementation of {@link AbstractResourceSubscriber} for handling catalog-related resource events.
 */
@Component("catalogResourceSubscriber")
@Slf4j
public class CatalogResourceSubscriber extends AbstractResourceSubscriber {

    /**
     * Messaging configuration properties for catalog-bs events.
     */
    private final MessagingCatalogBsProperties messagingCatalogBsProperties;

    /**
     * The client used to interact with the search dm.
     */
    private final SearchDmClient searchDmClient;

    /**
     * Constructs the CatalogResourceSubscriber
     *
     * @param searchBsConfigurationProperties the application configuration properties
     * @param searchDmClient                  http client for interacting with the search DM service
     */
    public CatalogResourceSubscriber(
            SearchBsConfigurationProperties searchBsConfigurationProperties,
            SearchDmClient searchDmClient
    ) {
        this.messagingCatalogBsProperties = searchBsConfigurationProperties.getMessaging().getCatalogBs();
        this.searchDmClient = searchDmClient;
    }

    /**
     * Processes a catalog resource event by invoking the appropriate Search DM operation.
     *
     * @param resourceId      the unique resource identifier
     * @param resourcePayload the resource data payload
     * @param eventType       the type of the event
     */
    @Override
    protected void handleResource(String resourceId, Object resourcePayload, String eventType) {
        log.debug("Processing catalog event, id: {}, payload: {}, type: {}", resourceId, resourcePayload, eventType);
        if (Objects.equals(eventType, messagingCatalogBsProperties.getResourceCreatedEventType()) ||
                Objects.equals(eventType, messagingCatalogBsProperties.getResourceUpdatedEventType())) {
            searchDmClient.upsertCatalog(resourceId, resourcePayload);
        } else if (Objects.equals(eventType, messagingCatalogBsProperties.getResourceDeletedEventType())) {
            searchDmClient.deleteCatalog(resourceId);
        } else {
            log.error("Unsupported eventType: {} for resourceId: {}", eventType, resourceId);
        }
    }
}
