package com.finconsgroup.itserr.marketplace.notificationfeeder.bs.event.extractor;

import com.fasterxml.jackson.databind.JsonNode;
import org.apache.commons.lang3.StringUtils;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * Component responsible for extracting field values from JSON data using path expressions. Supports JSON path-like syntax starting with '$.' prefix followed by
 * field names separated by dots.
 */
@Component
@Order(Ordered.HIGHEST_PRECEDENCE)
public class JsonFieldExtractor implements FieldExtractor {

    /** The prefix used in field path expressions to indicate JSON path syntax */
    public static final String PREFIX = "$.";
    /** Length of the PREFIX constant for optimization purposes */
    public static final int PREFIX_LENGTH = PREFIX.length();

    /**
     * Extracts field values from a JSON source using the specified field path.
     *
     * @param field The field path expression starting with '$.' prefix
     * @param source The source JSON object to extract values from
     * @return List of extracted field values as strings
     * @throws UnsupportedOperationException if source is not a JsonNode or field format is invalid
     */
    @Override
    @NonNull
    public List<String> extract(
            @NonNull final String field,
            @NonNull final Object source) {

        if (!(source instanceof JsonNode root)) {
            throw new UnsupportedOperationException("Source object should be a JsonNode instance");
        }

        if (!StringUtils.startsWith(field, PREFIX)) {
            throw new UnsupportedOperationException("Field format not supported");
        }

        final String[] fields = StringUtils.split(
                StringUtils.substring(field, PREFIX_LENGTH),
                ".");

        return extract(root, fields).stream()
                .map(JsonNode::asText)
                .toList();

    }

    /**
     * Recursively extracts field values from a JSON node using an array of field names.
     *
     * @param root The root JSON node to start extraction from
     * @param fields Array of field names representing the path to traverse
     * @return List of JSON nodes containing the extracted values
     */
    private List<JsonNode> extract(
            final JsonNode root,
            final String[] fields) {

        List<JsonNode> current = List.of(root);

        for (final String f : fields) {

            final List<JsonNode> next = new ArrayList<>();
            for (final JsonNode node : current) {

                final JsonNode child = node.get(f);
                if (child != null) {
                    if (child.isArray()) {
                        child.forEach(next::add);
                    } else {
                        next.add(child);
                    }
                }
            }

            current = next;
            if (current.isEmpty()) {
                break;
            }

        }

        return current;

    }

}
