package com.finconsgroup.itserr.marketplace.notification.dm.mapper;

import com.finconsgroup.itserr.marketplace.notification.dm.dto.InputCreateUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.InputPatchUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.OutputPatchUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.OutputUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.entity.UserNotificationEntity;
import org.jetbrains.annotations.NotNull;
import org.mapstruct.AfterMapping;
import org.mapstruct.BeanMapping;
import org.mapstruct.Mapper;
import org.mapstruct.Mapping;
import org.mapstruct.MappingTarget;
import org.mapstruct.NullValuePropertyMappingStrategy;
import org.springframework.lang.NonNull;

import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

/**
 * Mapper interface responsible for converting between UserNotification DTOs and entities.
 */
@Mapper(config = MapperConfiguration.class)
public interface UserNotificationMapper {

    /**
     * Maps an {@link InputCreateUserNotificationDto} to a {@link UserNotificationEntity} without setting the user field.
     *
     * @param src the source DTO to map from
     * @return the mapped entity with ignored fields
     */
    @Mapping(target = "id", ignore = true)
    @Mapping(target = "creationTime", ignore = true)
    @Mapping(target = "updateTime", ignore = true)
    @Mapping(target = "version", ignore = true)
    @Mapping(target = "read", ignore = true)
    @Mapping(target = "archived", ignore = true)
    @Mapping(target = "lastReadTime", ignore = true)
    @Mapping(target = "user", ignore = true)
    UserNotificationEntity toEntityWithoutUser(InputCreateUserNotificationDto src);

    /**
     * Converts a single notification DTO into multiple entities based on the list of users.
     *
     * @param src the source DTO containing user list and notification details
     * @return list of notification entities, one per user
     */
    @NonNull
    default List<UserNotificationEntity> toEntities(final InputCreateUserNotificationDto src) {

        if (src == null || src.getUsers() == null || src.getUsers().isEmpty()) {
            return new ArrayList<>();
        }

        final Set<String> users = new LinkedHashSet<>(src.getUsers());

        return users.stream()
                .map(user -> {
                    final UserNotificationEntity entity = toEntityWithoutUser(src);
                    entity.setUser(user);
                    return entity;
                })
                .toList();

    }

    /**
     * Maps a {@link UserNotificationEntity} to an {@link OutputUserNotificationDto}, using the provided contextual message for the "message" property.
     *
     * @param userNotificationEntity the source user notification entity to map
     * @return the mapped {@link OutputUserNotificationDto} containing the transformed data
     */
    OutputUserNotificationDto toDto(
            UserNotificationEntity userNotificationEntity);

    /**
     * Updates an existing entity with non-null values from the patch DTO.
     *
     * @param dto the DTO containing update values
     * @param entity the entity to update
     */
    @Mapping(target = "id", ignore = true)
    @Mapping(target = "creationTime", ignore = true)
    @Mapping(target = "updateTime", ignore = true)
    @Mapping(target = "version", ignore = true)
    @Mapping(target = "read", source = "read")
    @Mapping(target = "archived", source = "archived")
    @BeanMapping(nullValuePropertyMappingStrategy = NullValuePropertyMappingStrategy.IGNORE, ignoreByDefault = true)
    void patchEntity(InputPatchUserNotificationDto dto, @MappingTarget UserNotificationEntity entity);

    /**
     * Converts a user notification entity to a patch response DTO.
     *
     * @param userNotificationEntity the entity to convert
     * @return the mapped patch response DTO
     */
    OutputPatchUserNotificationDto toPatchDto(UserNotificationEntity userNotificationEntity);

    /**
     * Performs post-mapping operations after patching an entity. Updates the lastReadTime when the notification is marked as read.
     *
     * @param dto the source patch DTO
     * @param entity the target entity being updated
     */
    @AfterMapping
    default void afterPatchEntity(@NotNull InputPatchUserNotificationDto dto, @MappingTarget UserNotificationEntity entity) {
        // Handle read status changes and timestamps
        if (Boolean.TRUE.equals(dto.getRead())) {
            entity.setLastReadTime(ZonedDateTime.now(ZoneOffset.UTC).truncatedTo(ChronoUnit.MICROS));
        }
    }
}
