package com.finconsgroup.itserr.marketplace.notification.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.Table;
import jakarta.persistence.Version;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;

/**
 * Entity class representing a notification template for a specific notification type and language.
 */
@Entity
@Table(name = "notification_template")
@Data
@SuperBuilder
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class NotificationTemplateEntity extends AbstractUUIDEntity {

    /**
     * The type of notification (e.g., "catalog-item-event-created", "catalog-item-event-status").
     */
    @Column(name = "notification_type", nullable = false, length = 100)
    private String notificationType;

    /**
     * The language code for the notification template (e.g., "en", "it", "fr"). Uses ISO 639-1 two-letter language codes.
     */
    @Column(name = "language_code", nullable = false, length = 5)
    private String languageCode;

    /**
     * The actual message template text that will be used for generating notifications. May contain placeholders that will be replaced with actual values when
     * sending notifications.
     */
    @Column(name = "message_text", nullable = false)
    private String text;

    /**
     * The timestamp when the notification was created.
     */
    @Column(name = "creation_time", nullable = false)
    private ZonedDateTime creationTime;

    /**
     * The timestamp when the notification was last updated.
     */
    @Column(name = "update_time", nullable = false)
    private ZonedDateTime updateTime;

    /**
     * The version field used for optimistic locking.
     * <p>
     * This value is automatically managed by JPA to detect concurrent updates. Each time the entity is updated, the version is incremented. If two transactions
     * try to update the same entity simultaneously, JPA will detect the conflict based on this version and throw an
     * {@link jakarta.persistence.OptimisticLockException}.
     */
    @Version
    @Column(name = "version", nullable = false)
    private long version;

    @Override
    public void prePersist() {
        super.prePersist();
        ZonedDateTime now = ZonedDateTime.now(ZoneOffset.UTC).truncatedTo(ChronoUnit.MICROS);
        if (creationTime == null) {
            creationTime = now;
        }
        if (updateTime == null) {
            updateTime = now;
        }
    }

}
