package com.finconsgroup.itserr.marketplace.news.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.news.dm.dto.InputCreateNewsDto;
import com.finconsgroup.itserr.marketplace.news.dm.dto.InputUpdateNewsDto;
import com.finconsgroup.itserr.marketplace.news.dm.dto.OutputNewsDto;
import com.finconsgroup.itserr.marketplace.news.dm.entity.ArchivedNewsEntity;
import com.finconsgroup.itserr.marketplace.news.dm.entity.NewsEntity;
import com.finconsgroup.itserr.marketplace.news.dm.mapper.ArchivedNewsMapper;
import com.finconsgroup.itserr.marketplace.news.dm.mapper.NewsMapper;
import com.finconsgroup.itserr.marketplace.news.dm.repository.ArchivedNewsRepository;
import com.finconsgroup.itserr.marketplace.news.dm.repository.NewsRepository;
import com.finconsgroup.itserr.marketplace.news.dm.service.NewsService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.UUID;

/**
 * Default implementation of {@link NewsService} to perform operations related to news resources
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultNewsService implements NewsService {

    private final NewsRepository newsRepository;
    private final NewsMapper newsMapper;

    private final ArchivedNewsRepository archivedNewsRepository;
    private final ArchivedNewsMapper archivedNewsMapper;

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputNewsDto> findAll(@NonNull Pageable pageable) {
        return newsRepository.findAll(pageable)
                .map(newsMapper::toDto);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public OutputNewsDto findById(@NonNull UUID newsId) {
        NewsEntity newsEntity = newsRepository.findById(newsId)
                .orElseThrow(() -> new WP2ResourceNotFoundException(newsId));
        return newsMapper.toDto(newsEntity);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputNewsDto create(@NonNull UUID userId, @NotNull InputCreateNewsDto inputCreateNewsDto) {
        NewsEntity newsEntity = newsMapper.toEntity(inputCreateNewsDto);
        newsEntity.setCreatorId(userId);
        return newsMapper.toDto(newsRepository.saveAndFlush(newsEntity));
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputNewsDto updateById(@NonNull UUID userId, @NotNull UUID newsId,
                                    @NotNull InputUpdateNewsDto inputUpdateNewsDto) {
        NewsEntity newsEntity = findByIdOrThrow(userId, newsId);
        newsMapper.updateEntity(inputUpdateNewsDto, newsEntity);
        NewsEntity savedNewsEntity = newsRepository.save(newsEntity);
        return newsMapper.toDto(savedNewsEntity);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public void deleteById(@NonNull UUID userId, @NotNull UUID newsId) {
        NewsEntity newsEntity = findByIdOrThrow(userId, newsId);
        ArchivedNewsEntity archivedNewsEntity = archivedNewsMapper.toArchivedEntity(newsEntity);
        archivedNewsRepository.save(archivedNewsEntity);
        newsRepository.deleteById(newsId);
    }

    private NewsEntity findByIdOrThrow(@NonNull UUID userId, @NonNull UUID newsId) {
        return newsRepository
                .findByCreatorIdAndId(userId, newsId)
                .orElseThrow(() -> new WP2ResourceNotFoundException(newsId));
    }
}
