package com.finconsgroup.itserr.marketplace.metrics.feeder.messaging;

import com.fasterxml.jackson.core.type.TypeReference;
import com.finconsgroup.itserr.marketplace.metrics.feeder.client.MetricsBsClient;
import com.finconsgroup.itserr.marketplace.metrics.feeder.config.properties.MetricsFeederConfigurationProperties;
import com.finconsgroup.itserr.marketplace.metrics.feeder.dto.discussionbs.DiscussionThreadCreatedMessageBodyDto;
import com.finconsgroup.itserr.marketplace.metrics.feeder.dto.metricbs.InputCreateMetricEventDto;
import com.finconsgroup.itserr.marketplace.metrics.feeder.dto.metricbs.MetricDtoType;
import io.cloudevents.CloudEvent;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

/**
 * Subscriber class that handles discussion events and converts them into metric events.
 */
@Slf4j
@Component("discussionEventConsumer")
public class DiscussionEventConsumer
        extends AbstractMetricEventConsumer<DiscussionThreadCreatedMessageBodyDto> {

    /**
     * Constructor.
     *
     * @param config Configuration properties for the metrics feeder
     * @param metricsBsClient Client for interacting with the metrics business service
     */
    public DiscussionEventConsumer(
            final MetricsFeederConfigurationProperties config,
            final MetricsBsClient metricsBsClient) {

        super(
                metricsBsClient,
                MetricDtoType.CATALOG_ITEM_COMMENT,
                config.getMessaging().getDiscussion().getThreadCreatedEventType(),
                new TypeReference<>() {
                });

    }

    @Override
    protected void fillMetricEvent(
            @NonNull final InputCreateMetricEventDto.InputCreateMetricEventDtoBuilder metricEventBuilder,
            @NonNull final DiscussionThreadCreatedMessageBodyDto payload,
            @NonNull final CloudEvent event) throws ValidationException {

        if (payload.getAdditionalData() == null) {
            throw new ValidationException("no additional data");
        } else if (StringUtils.isBlank(payload.getAdditionalData().getDiscussionResourceId())) {
            throw new ValidationException("no discussion resource id");
        } else if (StringUtils.isBlank(payload.getAdditionalData().getDiscussionResourceType())) {
            throw new ValidationException("no discussion resource type");
        } else if (!StringUtils.equalsIgnoreCase(payload.getAdditionalData().getDiscussionResourceType(), "item")) {
            throw new ValidationException("not a catalog item discussion (type=" + payload.getAdditionalData().getDiscussionResourceType() + ")", false);
        }

        final String itemId = payload.getAdditionalData().getDiscussionResourceId();

        metricEventBuilder
                .resourceId(itemId);

    }

}
