package com.finconsgroup.itserr.marketplace.metrics.feeder.messaging;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import com.finconsgroup.itserr.marketplace.metrics.feeder.client.MetricsBsClient;
import com.finconsgroup.itserr.marketplace.metrics.feeder.dto.metricbs.InputCreateMetricEventDto;
import com.finconsgroup.itserr.marketplace.metrics.feeder.dto.metricbs.MetricDtoType;
import com.finconsgroup.itserr.marketplace.metrics.feeder.dto.metricbs.OutputMetricEventDto;
import com.finconsgroup.itserr.messaging.consumer.AbstractSimpleCloudEventConsumer;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;
import feign.FeignException;
import io.cloudevents.CloudEvent;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.NonNull;

import java.nio.ByteBuffer;
import java.time.Instant;
import java.time.ZoneOffset;
import java.util.Optional;
import java.util.UUID;

/**
 * Abstract base class for consumers that process events and convert them into metric events. Handles common validation and error handling logic for metric
 * event creation.
 *
 * @param <T> The type of messaging event DTO this consumer handles
 */
@Slf4j
public abstract class AbstractMetricEventConsumer<T extends MessagingEventDto<?>>
        extends AbstractSimpleCloudEventConsumer<T> {

    /** Client for interacting with the metrics business service. */
    private final MetricsBsClient metricsBsClient;

    /** The type of metric to be created. */
    private final MetricDtoType metricType;
    /** The type of event being handled, as defined in the source messaging system. */
    private final String eventType;

    /**
     * Exception thrown when event validation fails.
     */
    protected static class ValidationException extends Exception {

        /** Whether to log at WARN level (true) or INFO level (false) */
        final boolean warnLevel;

        /**
         * Creates a validation exception with warning level logging.
         *
         * @param message The validation error message
         */
        public ValidationException(String message) {
            this(message, true);
        }

        /**
         * Creates a validation exception with the specified logging level.
         *
         * @param message The validation error message
         * @param warnLevel If true, logs at WARN level, otherwise at INFO level
         */
        public ValidationException(String message, boolean warnLevel) {
            super(message);
            this.warnLevel = warnLevel;
        }

    }

    /**
     * Creates a new metric event consumer.
     *
     * @param metricsBsClient Client for interacting with metrics business service
     * @param metricType Type of metric this consumer creates
     * @param eventType Type of event this consumer handles
     * @param typeReference Type reference for deserializing event payload
     */
    public AbstractMetricEventConsumer(
            @NonNull final MetricsBsClient metricsBsClient,
            @NonNull final MetricDtoType metricType,
            @NonNull final String eventType,
            @NonNull final TypeReference<T> typeReference) {

        super(typeReference);

        this.metricType = metricType;
        this.eventType = eventType;
        this.metricsBsClient = metricsBsClient;

    }

    /**
     * Fills metric event details from the source event. Implementations should validate the payload and populate the metric event builder.
     *
     * @param metricEventBuilder Builder for creating the metric event
     * @param resourcePayload Payload from the source event
     * @param event The original cloud event
     * @throws ValidationException if the event data is invalid
     */
    protected abstract void fillMetricEvent(
            @NonNull InputCreateMetricEventDto.InputCreateMetricEventDtoBuilder metricEventBuilder,
            @NonNull T resourcePayload,
            @NonNull CloudEvent event) throws ValidationException;

    @Override
    protected void handleEvent(
            final T resourcePayload,
            @NonNull final CloudEvent event) {

        if (resourcePayload == null) {
            log.warn("Received {} event with no payload. Message will be dropped.", eventType);
        } else if (resourcePayload.getTimestamp() == null) {
            log.warn("Received {} event with no timestamp. Message will be dropped.", eventType);
        } else if (resourcePayload.getUser() == null) {
            log.warn("Received {} event with no user. Message will be dropped.", eventType);
        } else if (resourcePayload.getUser().getId() == null) {
            log.warn("Received {} event with no user id. Message will be dropped.", eventType);
        } else {

            try {

                final UUID userId = resourcePayload.getUser().getId();
                final Instant eventTime = resourcePayload.getTimestamp();

                final InputCreateMetricEventDto.InputCreateMetricEventDtoBuilder metricEventBuilder = InputCreateMetricEventDto.builder()
                        .eventTime(eventTime.atZone(ZoneOffset.UTC))
                        .eventAuthor(userId.toString());

                fillMetricEvent(metricEventBuilder, resourcePayload, event);

                final InputCreateMetricEventDto metricEvent = metricEventBuilder.build();

                final OutputMetricEventDto createdMetricEvent = this.metricsBsClient
                        .createMetricEvent(
                                metricType,
                                metricEvent);
                log.debug("Created metric event: {}", createdMetricEvent.getId());

            } catch (final ValidationException e) {

                if (e.warnLevel) {
                    log.warn("Invalid metric event {}: {}. Skipping creation.", eventType, e.getMessage());
                } else {
                    log.info("Invalid metric event {}: {}. Skipping creation.", eventType, e.getMessage());
                }

            } catch (final FeignException.FeignClientException.Conflict e) {

                log.info("Metric event already exists. Skipping creation.");

            } catch (final FeignException.FeignClientException.UnprocessableEntity e) {

                final String errorMessage = extractFirstErrorMessage(e);
                log.info("Event cannot be created: {}. Skipping creation.", errorMessage);

            }

        }

    }

    private static String extractFirstErrorMessage(final FeignException.FeignClientException e) {

        // Try to extract the error message from the response body
        final Optional<ByteBuffer> responseBodyOpt = e.responseBody();
        if (responseBodyOpt.isPresent()) {

            try {
                final ObjectMapper mapper = new ObjectMapper();
                final ErrorResponseDto errorResponse = mapper.readValue(
                        new String(responseBodyOpt.get().array()),
                        ErrorResponseDto.class);
                if (errorResponse != null
                        && errorResponse.messages() != null
                        && !errorResponse.messages().isEmpty()) {
                    return errorResponse.messages().getFirst();
                }

            } catch (final Exception parseException) {
                // Ignore
            }

        }

        // Fallback
        return e.getMessage();

    }

}
