package com.finconsgroup.itserr.marketplace.metrics.bs.repository;

import com.finconsgroup.itserr.marketplace.catalog.bs.client.ItemClient;
import com.finconsgroup.itserr.marketplace.catalog.bs.dto.ItemStatus;
import com.finconsgroup.itserr.marketplace.catalog.bs.dto.OutputItemDto;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.OutputCatalogItemMinimalDto;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import feign.FeignException;
import lombok.RequiredArgsConstructor;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * Repository class for handling catalog-related operations and data access. Provides methods to retrieve and manage catalog items through the ItemClient.
 */
@Component
@RequiredArgsConstructor
public class CatalogRepository {

    private final ItemClient itemClient;

    /**
     * Retrieves a catalog item by its unique identifier. If the item is not found, an empty {@code Optional} is returned.
     *
     * @param id the unique identifier of the catalog item to retrieve
     * @return an {@link Optional} containing the catalog item if found, or empty if not found
     */
    public Optional<OutputItemDto> findById(@NonNull final UUID id) {
        try {
            return Optional.of(itemClient.getItemById(id));
        } catch (final FeignException.NotFound | WP2ResourceNotFoundException e) {
            return Optional.empty();
        }
    }

    /**
     * Retrieves a set of item IDs for which the user is either an author or maintainer.
     *
     * @param userId the unique identifier of the user
     * @return a Set of item IDs as strings, or empty set if no items found
     */
    @NonNull
    public Set<String> getUserItemIds(@NonNull UUID userId) {
        Set<String> itemIds = itemClient.getAllItems(
                        0, Integer.MAX_VALUE, null, null,
                        null, null, null, userId,
                        ItemStatus.APPROVED, null)
                .getContent()
                .stream()
                .map(OutputItemDto::getId)
                .filter(Objects::nonNull)
                .map(UUID::toString)
                .collect(Collectors.toSet());
        return itemIds;
    }

    /**
     * Retrieves a set of item dto with minimal details for which the user is either an author or maintainer.
     *
     * @param userId the unique identifier of the user
     * @return a List of {@link OutputCatalogItemMinimalDto}, or empty list if no items found
     */
    @NonNull
    public List<OutputCatalogItemMinimalDto> getUserItems(@NonNull UUID userId) {
        List<OutputCatalogItemMinimalDto> items = itemClient.getAllItems(
                        0, Integer.MAX_VALUE, null, null,
                        null, null, null, userId,
                        ItemStatus.APPROVED, null)
                .getContent()
                .stream()
                .map(this::mapToOutputItemMinimalDto)
                .toList();
        return items;
    }

    private OutputCatalogItemMinimalDto mapToOutputItemMinimalDto(OutputItemDto outputItemDto) {
        return OutputCatalogItemMinimalDto.builder()
                .id(outputItemDto.getId().toString())
                .title(outputItemDto.getTitle())
                .build();
    }

}
