package com.finconsgroup.itserr.marketplace.metadata.dm.authorization;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2AuthorizationException;
import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.metadata.dm.config.SecurityConfig;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;

import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.UUID;

@Aspect
@Component
@Order(1)
@Slf4j
@RequiredArgsConstructor
public class RolesAspect {
    private final SecurityConfig securityConfig;

    @Around("@annotation(hasRoles)")
    public Object checkRoles(ProceedingJoinPoint pjp, HasRoles hasRoles) throws Throwable {
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        Set<String> userRoles = JwtTokenHolder.getRoles(
                securityConfig.getAudience()
        );

        String[] requiredRoles = hasRoles.value();

        String methodName = pjp.getSignature().toShortString();
        log.debug("@HasRoles check on {}: user={}, requiredRoles={}",
                methodName, userId, Arrays.toString(requiredRoles));

        boolean hasAllRoles = userRoles.containsAll(List.of(requiredRoles));

        if (!hasAllRoles) {
            log.warn("Authorization DENIED on {}: user={}, requiredRoles={}, userRoles={}",
                    methodName, userId, Arrays.toString(requiredRoles), userRoles);

            throw new WP2AuthorizationException(
                    String.format("User %s does not have the required roles: %s", userId, List.of(requiredRoles))
            );
        }

        log.debug("Authorization GRANTED on {}: user={}", methodName, userId);
        return pjp.proceed();
    }
}
