package com.finconsgroup.itserr.marketplace.metadata.dm.service;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.InputCreateMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.InputInternalFindMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.InputUpdateMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.OutputMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.MetadataStatus;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.OutputMetadataFieldDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.OutputMetadataFieldExtDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.OutputMetadataPreviewDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.entity.enumerated.MetadataCategoryEnum;
import com.finconsgroup.itserr.marketplace.metadata.dm.exception.MetadataNotFoundException;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;

import java.util.List;
import java.util.Optional;
import java.util.UUID;

/**
 * Service for handling business logic related to metadata.
 * This interface provides methods to retrieve, create, and delete metadata.
 *
 * <p>Example usage:
 * <pre>
 * OutputMetadataDto metadata = metadataService.findById(12);
 * </pre>
 * </p>
 */
public interface MetadataService {

    /**
     * Retrieves a paginated list of all metadata previews.
     *
     * @param category the category of the metadata
     * @param pageable the pagination information
     * @return a page of OutputMetadataPreviewDto representing the metadata
     */
    @NonNull
    Page<OutputMetadataPreviewDto> findAll(MetadataCategoryEnum category, Optional<UUID> userId, @NonNull Pageable pageable);

    /**
     * Retrieves a paginated list of all metadata by status and user.
     *
     * @param status    the status of metadata
     * @param creatorId the creator of metadata
     * @param pageable  the pagination information
     * @return a page of OutputMetadataPreviewDto representing the metadata
     */
    @NonNull
    OutputPageDto<OutputMetadataDto> findAllByCreatorId(MetadataStatus status, @NonNull UUID creatorId, @NonNull Pageable pageable);

    /**
     * Retrieves a paginated list of all metadata fields by Id.
     *
     * @param metadataId the id of the metadata
     * @param pageable   the pagination information
     * @return a page of OutputMetadataFieldDto representing the metadata
     */
    @NonNull
    Page<OutputMetadataFieldDto> findAllFieldsById(UUID metadataId, @NonNull Pageable pageable);

    /**
     * Retrieves a metadata by id.
     *
     * @param metadataId the id of the metadata to retrieve
     * @return an OutputMetadataDto representing the metadata details
     * @throws MetadataNotFoundException if no metadata is found with the given id
     */
    @NonNull
    OutputMetadataDto findById(@NonNull UUID metadataId);

    /**
     * Creates a new metadata.
     *
     * @param request Creation request. Cannot be null.
     * @return Created metadata. Never null.
     */
    @NonNull
    OutputMetadataDto create(@NonNull InputCreateMetadataDto request, @NonNull UUID userId);

    /**
     * Updates an existing metadata.
     *
     * @param metadataId             The unique identifier of the metadata to update.
     * @param inputUpdateMetadataDto The DTO containing the updated information.
     * @param userId                 the user id
     * @return the updated metadata
     */
    @NonNull
    OutputMetadataDto update(@NonNull UUID metadataId, @NonNull InputUpdateMetadataDto inputUpdateMetadataDto, @NonNull UUID userId);

    /**
     * Finds metadata according to criteria and return a list of full-detailed matching metadata.
     *
     * @param request Criteria. May be null to find all.
     * @return Found metadata. Empty list if no match.
     */
    @NonNull
    List<OutputMetadataDto> find(@Nullable InputInternalFindMetadataDto request);

    /**
     * Deletes a metadata by id.
     * It will move the metadata entity on a separated table containing deleted records.
     *
     * @param metadataId the id of the metadata to delete
     * @return an OutputMetadataDto representing the metadata details
     * @throws MetadataNotFoundException if no metadata is found with the given id
     */
    @NonNull
    OutputMetadataDto deleteById(@NonNull UUID metadataId);

    /**
     * Retrieves a paginated list of all metadata fields.
     *
     * @param category the category of the metadata
     * @return an OutputMetadataFieldExtDto representing the metadata field details
     * @throws MetadataNotFoundException if no metadata is found with the given id
     */
    @NonNull
    Page<OutputMetadataFieldExtDto> findAllFields(MetadataCategoryEnum category, Pageable pageable);

    /**
     * Requests moderation for a specific metadata.
     *
     * @param metadataId the unique identifier of the metadata to be moderated
     * @param userId     the unique identifier of the user requesting moderation
     * @return the moderation dto
     */
    @NonNull
    OutputMetadataDto requestModeration(UUID metadataId, UUID userId);
}
