package com.finconsgroup.itserr.marketplace.metadata.dm.repository.specification;

import com.finconsgroup.itserr.marketplace.metadata.dm.dto.MetadataStatus;
import com.finconsgroup.itserr.marketplace.metadata.dm.entity.MetadataEntity;
import com.finconsgroup.itserr.marketplace.metadata.dm.entity.MetadataFieldEntity;
import com.finconsgroup.itserr.marketplace.metadata.dm.entity.enumerated.MetadataCategoryEnum;
import jakarta.persistence.criteria.JoinType;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;

import java.util.Set;
import java.util.UUID;

/**
 * {@link MetadataFieldEntity} specifications.
 */

/**
 * {@link MetadataFieldEntity} specifications.
 */
public final class MetadataFieldSpecifications {


    /**
     * Private constructor.
     */
    private MetadataFieldSpecifications() {
        throw new UnsupportedOperationException("Cannot be instantiated");
    }

    /**
     * <p>Adds a fetch join to {@link MetadataEntity} for {@link MetadataFieldEntity} queries.</p>
     * <p>Fetch joins must not be applied to count queries generated for pagination.</p>
     *
     * @return a {@link Specification} that fetch-joins the metadata relation for field queries
     */
    @NonNull
    public static Specification<MetadataFieldEntity> fetchMetadata() {
        return (root, query, cb) -> {
            if (query == null) {
                return cb.conjunction();
            }
            if (query.getResultType() != Long.class && query.getResultType() != long.class) {
                root.fetch("metadata", JoinType.INNER);
                query.distinct(true);
            }
            return cb.conjunction();
        };
    }

    /**
     * <p>Creates a {@link Specification} to find {@link MetadataFieldEntity} by the associated metadata category.</p>
     *
     * @param category Metadata category, or null.
     * @return a {@link Specification} matching the associated metadata category
     */
    @NonNull
    public static Specification<MetadataFieldEntity> metadataHasCategory(@Nullable MetadataCategoryEnum category) {
        return (root, query, cb) ->
                category == null
                        ? cb.conjunction()
                        : cb.equal(root.join("metadata").get("category"), category);
    }

    /**
     * <p>Creates a {@link Specification} to exclude {@link MetadataFieldEntity} by the associated metadata category.</p>
     *
     * @param category Metadata category to exclude (not null)
     * @return a {@link Specification} excluding the associated metadata category
     */
    @NonNull
    public static Specification<MetadataFieldEntity> metadataExcludeCategory(@NonNull MetadataCategoryEnum category) {
        return (root, query, cb) -> cb.notEqual(root.join("metadata").get("category"), category);
    }

    /**
     * <p>Creates a {@link Specification} to find {@link MetadataFieldEntity} by the associated metadata statuses.</p>
     *
     * @param statuses Metadata statuses, or null.
     * @return a {@link Specification} matching the associated metadata statuses
     */
    @NonNull
    public static Specification<MetadataFieldEntity> metadataHasStatusIn(@Nullable Set<MetadataStatus> statuses) {
        return (root, query, cb) ->
                statuses == null || statuses.isEmpty()
                        ? cb.conjunction()
                        : root.join("metadata").get("status").in(statuses);
    }

    /**
     * <p>Creates a {@link Specification} to find {@link MetadataFieldEntity} by the associated metadata creatorId.</p>
     *
     * @param creatorId Metadata creatorId, or null.
     * @return a {@link Specification} matching the associated metadata creatorId
     */
    @NonNull
    public static Specification<MetadataFieldEntity> metadataHasCreatorId(@Nullable UUID creatorId) {
        return (root, query, cb) ->
                creatorId == null
                        ? cb.conjunction()
                        : cb.equal(root.join("metadata").get("creatorId"), creatorId);
    }
}
