package com.finconsgroup.itserr.marketplace.metadata.bs.messaging.impl;

import com.finconsgroup.itserr.marketplace.metadata.bs.config.MetadataBsConfigurationProperties;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.OutputMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.messaging.ResourceProducer;
import com.finconsgroup.itserr.messaging.exception.WP2MessagingException;
import com.finconsgroup.itserr.messaging.producer.MessageProducer;
import com.finconsgroup.itserr.messaging.producer.ProducerRegistry;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

import java.util.Map;

import static com.finconsgroup.itserr.marketplace.metadata.bs.constants.Constants.SERVICE_FILTER_KEY;
import static com.finconsgroup.itserr.marketplace.metadata.bs.constants.Constants.SERVICE_FILTER_VALUE;

/**
 * Default implementation of {@link ResourceProducer} that publishes
 * resource events (create, update, delete) for catalog items.
 */
@Component
@ConditionalOnProperty(name = "messaging.enabled", havingValue = "true")
@Slf4j
public class DefaultResourceProducer implements ResourceProducer {

    private final MessageProducer messageProducer;

    private final MetadataBsConfigurationProperties.Messaging messagingProperties;

    /**
     * Constructs the DefaultResourceProducer.
     *
     * @param metadataBsConfigurationProperties application-specific configuration properties
     * @param producerRegistry                  the registry providing message producers
     */
    public DefaultResourceProducer(
            MetadataBsConfigurationProperties metadataBsConfigurationProperties,
            ProducerRegistry producerRegistry
    ) {
        this.messagingProperties = metadataBsConfigurationProperties.getMessaging();
        this.messageProducer = producerRegistry.getMessageProducer(this.messagingProperties.getProducerName());
    }

    @Override
    public void publishApprovedResource(@NonNull OutputMetadataDto resource) {
        publish(resource, messagingProperties.getResourceCreatedType());
    }

    @Override
    public void publishDeletedResource(@NonNull OutputMetadataDto resource) {
        publish(resource, messagingProperties.getResourceDeletedType());
    }

    /**
     * Sends a message to the messaging system.
     *
     * @param resource  the payload of the message
     * @param eventType the type of event to publish
     * @throws WP2MessagingException if message publishing fails
     */
    private void publish(@NonNull OutputMetadataDto resource, @NonNull String eventType) {
        try {
            Map<String, OutputMetadataDto> messageMap = Map.of(resource.getId().toString(), resource);
            Map<String, String> filterProperties = Map.of(SERVICE_FILTER_KEY, SERVICE_FILTER_VALUE);
            log.debug("Sending message, messageMap: %s, eventType: %s, source: %s, filterProperties: %s"
                    .formatted(messageMap, eventType, messagingProperties.getSource(), filterProperties));
            messageProducer.publishEvent(messageMap, eventType, messagingProperties.getSource(), filterProperties);
            log.info("Successfully published message for resource with id: %s".formatted(resource.getId()));
        } catch (Exception e) {
            String errorMessage = "Error occurred while sending message: %s".formatted(e.getMessage());
            throw new WP2MessagingException(errorMessage, e);
        }
    }
}
