package com.finconsgroup.itserr.marketplace.metadata.bs.config;

import jakarta.validation.Valid;
import jakarta.validation.constraints.NotBlank;
import jakarta.validation.constraints.NotNull;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.validation.annotation.Validated;

/**
 * Configuration properties for the metadata bs service.
 * <p>
 * These properties are mapped from the {@code metadata.bs} prefix in the application configuration.
 */
@Data
@Validated
@ConfigurationProperties(prefix = "metadata.bs")
public class MetadataBsConfigurationProperties {

    /**
     * Base URL used by Swagger/OpenAPI documentation.
     * Example: {@code http://localhost:8080}
     */
    @NotBlank
    private String swaggerApiServerUrl;

    /**
     * Messaging-related configuration properties.
     */
    @NotNull
    @Valid
    private Messaging messaging;

    /**
     * Diagnostics-related configuration properties.
     */
    @NotNull
    @Valid
    private Diagnostics diagnostics;

    /**
     * Configuration for messaging
     */
    @Data
    @NoArgsConstructor
    @AllArgsConstructor
    public static class Messaging {

        /**
         * Name of the configured producer
         */
        @NotBlank
        private String producerName;

        /**
         * Name of the configured event producer
         */
        @NotBlank
        private String eventProducerName;

        /**
         * Source identifier for events, used to indicate the origin of a message.
         */
        @NotBlank
        private String source;

        /**
         * Event type string used for resource creation.
         */
        @NotBlank
        private String resourceCreatedType;

        /**
         * Event type string used for resource deletion.
         */
        @NotBlank
        private String resourceDeletedType;

        /**
         * Messaging Event type string used for resource event creation.
         */
        @NotBlank
        private String resourceCreatedEventType;

        /**
         * Messaging Event type string used for resource event update.
         */
        @NotBlank
        private String resourceUpdatedEventType;

        /**
         * Messaging Event type string used for moderation request for resource event.
         */
        @NotBlank
        private String resourceModerationRequested;

        /**
         * Messaging Event type string used for resource event deletion.
         */
        @NotBlank
        private String resourceDeletedEventType;

        /**
         * Messaging Event type string used for resource event status.
         */
        @NotBlank
        private String resourceStatusEventType;

    }

    /**
     * Configuration properties for diagnostics-related features.
     */
    @Data
    @NoArgsConstructor
    @AllArgsConstructor
    public static class Diagnostics {
        /**
         * Controls whether the publish message diagnostic endpoint is enabled.
         */
        private boolean publishMessageEnabled;
    }
}
