package com.finconsgroup.itserr.marketplace.metadata.bs.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.MetadataDmClient;
import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.dto.InputCreateMetadataDmDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.dto.InputUpdateMetadataDmDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.dto.MetadataCategoryDmEnum;
import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.dto.MetadataDmStatus;
import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.dto.OutputMetadataDmDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.dto.OutputMetadataFieldDmDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.dto.OutputMetadataFieldExtDmDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.dto.OutputMetadataPreviewDmDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.InputCreateMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.InputUpdateMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.MetadataCategoryEnum;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.MetadataStatus;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.OutputMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.OutputMetadataFieldDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.OutputMetadataFieldExtDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.OutputMetadataPreviewDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.exception.MetadataExistsException;
import com.finconsgroup.itserr.marketplace.metadata.bs.exception.WP2ServiceUnavailableException;
import com.finconsgroup.itserr.marketplace.metadata.bs.mapper.MetadataMapper;
import com.finconsgroup.itserr.marketplace.metadata.bs.messaging.ResourceProducer;
import com.finconsgroup.itserr.marketplace.metadata.bs.service.MetadataService;
import feign.FeignException;
import lombok.RequiredArgsConstructor;
import org.jetbrains.annotations.NotNull;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Sort;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;
import java.util.UUID;

/**
 * Default implementation of a {@link MetadataService}.
 */
@Service
@RequiredArgsConstructor
public class DefaultMetadataService implements MetadataService {

    /**
     * Metadata mapper.
     */
    private final MetadataMapper metadataMapper;

    private final MetadataDmClient metadataDmClient;

    private final ResourceProducer resourceProducer;

    private final UserProfileHelper userProfileHelper;

    @Override
    @NonNull
    public OutputPageDto<OutputMetadataPreviewDto> findAll(MetadataCategoryEnum category, int pageNumber, int pageSize, String sort, Sort.Direction direction) {
        MetadataCategoryDmEnum metadataCategoryDmEnum = (category != null) ? MetadataCategoryDmEnum.fromString(category.getValue()) : null;
        Page<OutputMetadataPreviewDmDto> outputMetadataDmDtoOutputPageDto = metadataDmClient.findAll(metadataCategoryDmEnum,
                pageNumber, pageSize, sort, direction);

        Map<UUID, OutputUserProfileDto> userDetails = userProfileHelper.fetchUserDetailsFromPreviews(outputMetadataDmDtoOutputPageDto.toList());

        return OutputPageDto.fromPage(outputMetadataDmDtoOutputPageDto)
                .map(dto -> {
                    OutputMetadataPreviewDto outputMetadataPreviewDto = metadataMapper.clientDtoToDto(dto);
                    userProfileHelper.enrichMetadataPreviewDto(dto, outputMetadataPreviewDto, userDetails);
                    return outputMetadataPreviewDto;
                });
    }

    @Override
    @NonNull
    public OutputMetadataDto findById(@NonNull UUID metadataId) {
        OutputMetadataDmDto outputMetadataDmDto = metadataDmClient.findById(metadataId);
        Map<UUID, OutputUserProfileDto> userDetails = userProfileHelper.fetchUserDetails(List.of(outputMetadataDmDto));
        OutputMetadataDto outputMetadataDto = metadataMapper.clientDtoToDto(outputMetadataDmDto);
        userProfileHelper.enrichMetadataDto(outputMetadataDmDto, outputMetadataDto, userDetails);
        return outputMetadataDto;
    }

    @Override
    @NonNull
    public OutputMetadataDto create(@NotNull final InputCreateMetadataDto request) {

        // Create metadata
        final OutputMetadataDto createdMetadata = executeCreateMetadata(request);

        // Send message into bus
        resourceProducer.publishCreatedResource(createdMetadata);

        // Return created metadata
        return createdMetadata;

    }

    @NonNull
    @Override
    public OutputMetadataDto update(@NonNull UUID metadataId, @NonNull InputUpdateMetadataDto inputUpdateMetadataDto) {
        final InputUpdateMetadataDmDto inputUpdateMetadataDmDto = metadataMapper.dtoToClientDto(inputUpdateMetadataDto);
        OutputMetadataDmDto outputMetadataDmDto = metadataDmClient.update(metadataId, inputUpdateMetadataDmDto);
        Map<UUID, OutputUserProfileDto> userDetails = userProfileHelper.fetchUserDetails(List.of(outputMetadataDmDto));
        OutputMetadataDto outputMetadataDto = metadataMapper.clientDtoToDto(outputMetadataDmDto);
        userProfileHelper.enrichMetadataDto(outputMetadataDmDto, outputMetadataDto, userDetails);
        return outputMetadataDto;
    }

    /**
     * Creates a metadata calling the DM service.
     *
     * @param request Request.
     * @return Created metadata.
     * @throws com.finconsgroup.itserr.marketplace.metadata.bs.exception.MetadataExistsException when a metadata with the same name already exists.
     * @throws WP2ServiceUnavailableException                                                    when downstream service is not available.
     */
    private OutputMetadataDto executeCreateMetadata(@NotNull InputCreateMetadataDto request) {

        // Map to client request
        final InputCreateMetadataDmDto clientRequest = metadataMapper.dtoToClientDto(request);

        // Execute
        final OutputMetadataDmDto createdMetadata;
        try {
            // Create metadata using remote service
            createdMetadata = metadataDmClient.create(clientRequest);
            Map<UUID, OutputUserProfileDto> userDetails = userProfileHelper.fetchUserDetails(List.of(createdMetadata));
            OutputMetadataDto outputMetadataDto = metadataMapper.clientDtoToDto(createdMetadata);
            userProfileHelper.enrichMetadataDto(createdMetadata, outputMetadataDto, userDetails);
            return outputMetadataDto;
        } catch (final FeignException.Conflict e) {
            // Another metadata exists with the same name
            throw new MetadataExistsException(request.getName());
        }
    }

    @NotNull
    @Override
    public OutputMetadataDto deleteById(@NotNull final UUID id) {

        // Delete metadata
        OutputMetadataDmDto outputMetadataDmDto = metadataDmClient.delete(id);
        Map<UUID, OutputUserProfileDto> userDetails = userProfileHelper.fetchUserDetails(List.of(outputMetadataDmDto));
        OutputMetadataDto outputMetadataDto = metadataMapper.clientDtoToDto(outputMetadataDmDto);
        userProfileHelper.enrichMetadataDto(outputMetadataDmDto, outputMetadataDto, userDetails);
        // Send message into bus
        resourceProducer.publishDeletedResource(outputMetadataDto.getId());

        // Return deleted metadata
        return outputMetadataDto;
    }

    @NonNull
    @Override
    public OutputPageDto<OutputMetadataFieldDto> findAllFieldsById(UUID metadataId, int pageNumber, int pageSize, String sort, Sort.Direction direction) {
        Page<OutputMetadataFieldDmDto> outputMetadataFieldDmOutputPageDto = metadataDmClient
                .findAllFieldsById(metadataId, pageNumber, pageSize, sort, direction);

        return OutputPageDto.fromPage(outputMetadataFieldDmOutputPageDto).map(metadataMapper::clientDtoToDto);
    }

    @NonNull
    @Override
    public OutputPageDto<OutputMetadataFieldExtDto> findAllFields(MetadataCategoryEnum category, int pageNumber, int pageSize, String sort, Sort.Direction direction) {
        MetadataCategoryDmEnum metadataCategoryDmEnum = (category != null) ? MetadataCategoryDmEnum.fromString(category.getValue()) : null;
        Page<OutputMetadataFieldExtDmDto> outputMetadataFieldExtDmDtoOutputPageDto = metadataDmClient.findAllFields(metadataCategoryDmEnum,
                pageNumber, pageSize, sort, direction);

        Map<UUID, OutputUserProfileDto> userDetails = userProfileHelper.fetchUserDetailsFromMetadataFieldExt(outputMetadataFieldExtDmDtoOutputPageDto.toList());

        return OutputPageDto.fromPage(outputMetadataFieldExtDmDtoOutputPageDto)
                .map(dto -> {
                    OutputMetadataFieldExtDto outputMetadataFieldExtDto = metadataMapper.clientDtoToDto(dto);
                    userProfileHelper.enrichMetadataFieldExtDto(dto, outputMetadataFieldExtDto, userDetails);
                    return outputMetadataFieldExtDto;
                });
    }

    @NonNull
    @Override
    public OutputMetadataDto requestModeration(@NonNull UUID metadataId) {
        OutputMetadataDmDto outputMetadataDmDto = metadataDmClient.requestModeration(metadataId);
        Map<UUID, OutputUserProfileDto> userDetails = userProfileHelper.fetchUserDetails(List.of(outputMetadataDmDto));
        OutputMetadataDto outputMetadataDto = metadataMapper.clientDtoToDto(outputMetadataDmDto);
        userProfileHelper.enrichMetadataDto(outputMetadataDmDto, outputMetadataDto, userDetails);
        return outputMetadataDto;
    }

    @NonNull
    @Override
    public OutputPageDto<OutputMetadataDto> findAllByCreatorId(MetadataStatus status, int pageNumber, int pageSize, String sort, Sort.Direction direction) {
        MetadataDmStatus metadataDmStatus = (status != null) ? MetadataDmStatus.fromString(status.getValue()) : null;
        OutputPageDto<OutputMetadataDmDto> outputMetadataDmDtoOutputPageDto = metadataDmClient.findAllByCreatorId(metadataDmStatus,
                pageNumber, pageSize, sort, direction);

        Map<UUID, OutputUserProfileDto> userDetails = userProfileHelper.fetchUserDetails(outputMetadataDmDtoOutputPageDto.getContent());

        List<OutputMetadataDto> outputMetadataDtoList = outputMetadataDmDtoOutputPageDto.getContent().stream()
                .map(dto -> {
                    OutputMetadataDto outputMetadataDto = metadataMapper.clientDtoToDto(dto);
                    userProfileHelper.enrichMetadataDto(dto, outputMetadataDto, userDetails);
                    return outputMetadataDto;
                })
                .toList();

        OutputPageDto<OutputMetadataDto> outputMetadataDtoOutputPageDto = OutputPageDto.emptyWithPage(outputMetadataDmDtoOutputPageDto.getPage());
        outputMetadataDtoOutputPageDto.setContent(outputMetadataDtoList);
        return outputMetadataDtoOutputPageDto;
    }
}
