package com.finconsgroup.itserr.marketplace.institutionalpage.dm.component;

import com.finconsgroup.itserr.marketplace.core.web.exception.MissingRoleException;
import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.core.web.security.jwt.SecurityRoles;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.InstitutionalPageEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.MemberEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.InstitutionalPageNotFoundException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.NotInstitutionalPageMemberException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.NotInstitutionalPageWPLeaderException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.UsersAlreadyMemberException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.model.MembersInHierarchyModel;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.repository.MemberRepository;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.UUID;

/**
 * Helper component for managing and verifying membership and leadership roles
 * within {@link InstitutionalPageEntity} instances.
 */
@Component
@RequiredArgsConstructor
public class MemberHelper {

    @Value("${institutional-page.dm.security.audience}")
    private String securityAudience;

    private final InstitutionalPageHelper institutionalPageHelper;
    private final MemberRepository memberRepository;

    // roles

    public void deleteRegularMembersFromDescendants(
            UUID userId,
            UUID rootInstitutionalPageId,
            List<UUID> targetUserIds
    ) {
        List<InstitutionalPageEntity> institutionalPages = institutionalPageHelper
                .retrieveAllOriginalInHierarchyForMemberByRootId(userId, Pageable.unpaged(), rootInstitutionalPageId)
                .getContent();
        List<InstitutionalPageEntity> updatedInstitutionalPages = new LinkedList<>();
        for (InstitutionalPageEntity institutionalPage : institutionalPages) {
            boolean removed = institutionalPage.getMembers()
                    .removeIf(member -> !member.isWpLead() && targetUserIds.contains(member.getMemberId()));
            if(removed){
                updatedInstitutionalPages.add(institutionalPage);
            }
        }
        institutionalPageHelper.saveAll(userId, updatedInstitutionalPages);
    }

    public void verifyIPModeratorOrWpLeaderOrThrow(UUID userId, UUID institutionalPageId) {
        if (hasIPModerateRole()) {
            return;
        }
        verifyWpLeaderOrThrow(userId, institutionalPageId);
    }

    public void verifyIPModeratorOrThrow() {
        if (!hasIPModerateRole()) {
            // TODO use proper role when available
            throw new MissingRoleException(JwtTokenHolder.getUserIdOrThrow(), Set.of(SecurityRoles.MEMBER_ROLE));
        }
    }

    public boolean hasIPModerateRole() {
        // TODO use proper role when available
        return JwtTokenHolder.getRoles(securityAudience).contains(SecurityRoles.MEMBER_ROLE);
    }

    public void verifyWpLeaderOrThrow(UUID userId, UUID institutionalPageId) {
        InstitutionalPageEntity institutionalPage = institutionalPageHelper.retrieveInstitutionalPage(institutionalPageId);
        if (!isWpLeader(userId, institutionalPage)) {
            if (isRegularMember(userId, institutionalPage) || institutionalPage.isPublished()) {
                throw new NotInstitutionalPageWPLeaderException(userId, institutionalPageId);
            } else {
                throw new InstitutionalPageNotFoundException(institutionalPageId);
            }
        }
    }

    public boolean isWpLeader(UUID userId, InstitutionalPageEntity institutionalPage) {
        UUID rootId = InstitutionalPageHelper.getRootInstitutionalPageId(institutionalPage);
        return memberRepository.existsByMemberIdAndInstitutionalPageIdAndWpLead(userId, rootId, true);
    }

    public boolean isMember(UUID userId, InstitutionalPageEntity institutionalPage) {
        return isRegularMember(userId, institutionalPage) || isWpLeader(userId, institutionalPage);
    }

    public boolean isRegularMember(UUID userId, InstitutionalPageEntity institutionalPage) {
        return memberRepository.existsByMemberIdAndInstitutionalPageIdAndWpLead(
                userId,
                institutionalPage.getId(),
                false
        );
    }

    public void verifyMemberOrThrow(UUID userId, UUID institutionalPageId) {
        InstitutionalPageEntity institutionalPage = institutionalPageHelper.retrieveInstitutionalPage(institutionalPageId);
        if (!isMember(userId, institutionalPage)) {
            if (institutionalPage.isPublished()) {
                throw new NotInstitutionalPageMemberException(userId, institutionalPageId);
            } else {
                throw new InstitutionalPageNotFoundException(institutionalPageId);
            }
        }
    }

    public void verifyNotAlreadyMemberOrThrow(UUID userId, UUID institutionalPageId) {
        InstitutionalPageEntity institutionalPage = institutionalPageHelper.retrieveInstitutionalPage(institutionalPageId);
        if (isMember(userId, institutionalPage)) {
            throw new UsersAlreadyMemberException(institutionalPageId, List.of(userId));
        }
    }

    public void verifyNotAlreadyMembersOrThrow(List<UUID> userIds, UUID institutionalPageId) {
        List<UUID> alreadyMembers = retrieveMembersByInstitutionalPageAndMemberIds(institutionalPageId, userIds)
                .stream()
                .map(memberEntity -> memberEntity.getMemberId())
                .toList();
        if (!alreadyMembers.isEmpty()) {
            throw new UsersAlreadyMemberException(institutionalPageId, alreadyMembers);
        }
    }

    // find users

    public MembersInHierarchyModel findUsersByRootId(Pageable pageable, UUID rootInstitutionalPageId) {
        // find page of user ids with proper size
        Page<UUID> userIdPage = memberRepository
                .findDistinctMemberIdsOrderedByWpLead(pageable, rootInstitutionalPageId)
                .map(membershipModel -> membershipModel.getMemberId());
        // find all members in rootInstitutionalPageId hierarchy for the userIdPage
        List<MemberEntity> members = memberRepository
                .findAllMembersByRootIdAndMemberIds(rootInstitutionalPageId, userIdPage.getContent());
        return MembersInHierarchyModel.builder()
                .members(members)
                .pageable(pageable)
                .totalElements(userIdPage.getTotalElements())
                .build();
    }

    // private

    private List<MemberEntity> retrieveWpLeadsByInstitutionalPageAndMemberIds(
            UUID institutionalPageId,
            List<UUID> memberIds
    ) {
        UUID rootId = institutionalPageHelper.retrieveRootInstitutionalPageId(institutionalPageId);
        return memberRepository.findAllByInstitutionalPageIdAndMemberIdInAndWpLead(rootId, memberIds, true);
    }

    private List<MemberEntity> retrieveMembersByInstitutionalPageAndMemberIds(UUID institutionalPageId, List<UUID> memberIds) {
        List<MemberEntity> memberEntities = retrieveWpLeadsByInstitutionalPageAndMemberIds(
                institutionalPageId,
                memberIds
        );
        memberEntities.addAll(memberRepository.findAllByInstitutionalPageIdAndMemberIdInAndWpLead(
                institutionalPageId,
                memberIds,
                false
        ));
        return memberEntities;
    }

    public List<MemberEntity> retrieveWPLeadersByInstitutionalPages(List<InstitutionalPageEntity> institutionalPages) {
        List<UUID> rootIds = institutionalPages.stream()
                .map(ip -> InstitutionalPageHelper.getRootInstitutionalPageId(ip))
                .toList();
        return memberRepository.findAllByInstitutionalPageIdInAndWpLead(rootIds, true);
    }

    public List<MemberEntity> retrieveRegularMembersByInstitutionalPages(
            List<InstitutionalPageEntity> institutionalPages
    ) {
        List<UUID> originalIds = institutionalPages.stream()
                .map(ip -> InstitutionalPageHelper.getOriginalInstitutionalPageId(ip))
                .toList();
        return memberRepository.findAllByInstitutionalPageIdInAndWpLead(originalIds, false);
    }

}
