package com.finconsgroup.itserr.marketplace.institutional_page.bs.service;

import com.finconsgroup.itserr.marketplace.core.web.enums.SortDirection;
import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputCreateInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputUpdateInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputSearchForMemberInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.ModerationStatus;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputRequestUpdateDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputWorkspaceItemDto;
import org.springframework.lang.NonNull;
import org.springframework.web.multipart.MultipartFile;

import java.util.List;
import java.util.Set;
import java.util.UUID;

public interface InstitutionalPageService {
    /**
     * Create an institutional page
     *
     * @param institutionalPageDto the dto representing the institutional page to create
     * @param imageFile optional image file to upload
     * @return the institutional page created
     */
    OutputInstitutionalPageDto createInstitutionalPage(InputCreateInstitutionalPageDto institutionalPageDto, MultipartFile imageFile);

    /**
     * Get all institutional pages for which the user is a contributor
     *
     * @param moderationStatus   the status filter to be applied (required).
     * @param includePublishedAndNotMember The flag to indicate if published institutional pages for which the user is NOT a member should be returned.
     * @param associationsToLoad the associations to be returned on the response
     * @param pageNumber the number of the page
     * @param pageSize the size of the page
     * @param sort the attribute to sort by
     * @param direction the direction of sorting
     * @return the institutional page
     */
    OutputPageDto<OutputInstitutionalPageDto> getAllInstitutionalPages(ModerationStatus moderationStatus, boolean includePublishedAndNotMember, Set<String> associationsToLoad, int pageNumber, int pageSize, String sort, SortDirection direction);

    /**
     * Get an institutional page by id
     *
     * @param institutionalPageId the id of the institutional page
     * @param moderationStatus   the status filter to be applied (required).
     * @param includePublishedAndNotMember The flag to indicate if published institutional pages for which the user is NOT a member should be returned.
     * @return the institutional page
     */
    OutputInstitutionalPageDto getInstitutionalPageById(UUID institutionalPageId, ModerationStatus moderationStatus, boolean includePublishedAndNotMember);

    /**
     * Update an institutional page
     *
     * @param institutionalPageDto the institutional page to update
     * @param institutionalPageId the id of the institutional page
     * @param imageFile optional image file to upload
     * @return the updated institutional page
     */
    OutputInstitutionalPageDto updateInstitutionalPage(UUID institutionalPageId, InputUpdateInstitutionalPageDto institutionalPageDto, MultipartFile imageFile);

    /**
     * Delete an institutional page
     *
     * @param institutionalPageId the id of the institutional page to delete
     */
    void deleteInstitutionalPage(UUID institutionalPageId);

    /**
     * Retrieves a paginated list of all InstitutionalPages matching the search criteria.
     *
     * @param moderationStatus the status filter to be applied (required).
     * @param inputSearchForMemberInstitutionalPageDto the input DTO containing search criteria
     * @param associationsToLoad the associations to be returned on the response
     * @param pageNumber                      the number of the page
     * @param pageSize                        the size of the page
     * @param sort                            the attribute to sort by
     * @param direction                       the direction of sorting
     * @return a page of OutputInstitutionalPageDto representing the InstitutionalPages
     */
    @NonNull
    OutputPageDto<OutputInstitutionalPageDto> search(ModerationStatus moderationStatus, @NonNull InputSearchForMemberInstitutionalPageDto inputSearchForMemberInstitutionalPageDto,
                                                     Set<String> associationsToLoad, int pageNumber, int pageSize,
                                                     String sort, SortDirection direction);

    /**
     * Retrieves a paginated list of all InstitutionalPages the user is contributing to,
     * corresponding to the hierarchy associated with the rootInstitutionalPageId.
     *
     * @param rootInstitutionalPageId ID of the root institutional page
     * @param associationsToLoad comma separated list of the associations to be returned (default is "all")
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize the number of InstitutionalPages per page (default is 10)
     * @param sort the field to sort by (default is "id")
     * @param direction the direction of sorting (default is ascending)
     * @return a page of {@link OutputInstitutionalPageDto} representing the institutional pages in the hierarchy
     */
    @NonNull
    OutputPageDto<OutputInstitutionalPageDto> findInstitutionalPagesHierarchyByRootId(
            @NonNull UUID rootInstitutionalPageId,
            Set<String> associationsToLoad,
            int pageNumber,
            int pageSize,
            String sort,
            SortDirection direction
    );

    /**
     * Retrieve folder content related to institutional page
     *
     * @param institutionalPageId the id of the institutional page
     * @param pageNumber          the number of the page
     * @param pageSize            the size of the page
     * @return a list of OutputWorkspaceItemDto representing the folder content
     */
    List<OutputWorkspaceItemDto> retrieveFolderContent(UUID institutionalPageId, int pageNumber, int pageSize);

    /**
     * Requests the moderation of an institutional page by its ID.
     *
     * @param institutionalPageId the UUID of the institutional page to be moderated
     * @return the details of the institutional page after the publication request
     */
    OutputInstitutionalPageDto requestModerationInstitutionalPage(UUID institutionalPageId);

    /**
     * Requests the publication of an institutional page by its ID.
     *
     * @param institutionalPageId the UUID of the institutional page to be published
     * @return the details of the institutional page after the publication request
     */
    OutputInstitutionalPageDto requestPublicationInstitutionalPage(UUID institutionalPageId);

    /**
     * Request update for a specific InstitutionalPage by id, only if the user is a member.
     *
     * @param institutionalPageId the id of the pending InstitutionalPage
     * @return a {@link OutputRequestUpdateDto} indicating whether the lock for update has been obtained
     */
    @NonNull
    OutputRequestUpdateDto requestUpdateInstitutionalPageById(@NonNull UUID institutionalPageId);

    /**
     * Release lock-for-update of a specific InstitutionalPage by id, only if the user is a member.
     *
     * @param institutionalPageId the id of the pending InstitutionalPage
     */
    void cancelUpdateInstitutionalPageById(@NonNull UUID institutionalPageId);

}
