package com.finconsgroup.itserr.marketplace.favourite.user.bs.messaging;

import com.fasterxml.jackson.core.type.TypeReference;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.config.FavouriteUserBsProperties;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.config.MessagingCatalogBsProperties;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.enums.ItemContext;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.messaging.dto.CatalogItemMessagingStatusAdditionalDataDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.service.FavouriteUserItemService;
import com.finconsgroup.itserr.messaging.consumer.CloudEventConsumer;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;
import com.finconsgroup.itserr.messaging.dto.UntypedMessagingEventDto;
import io.cloudevents.CloudEvent;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

@Component("catalogEventConsumer")
@Slf4j
public class CatalogEventConsumer extends CloudEventConsumer {

    private static final String OPERATION_TYPE_CREATE = "create";
    private static final String STATUS_APPROVED = "approved";

    private final FavouriteUserItemService favouriteUserItemService;

    /**
     * Constructs the CatalogEventConsumer
     *
     * @param favouriteUserBsConfigurationProperties the application configuration properties
     * @param favouriteUserItemService               favourite user item service
     */
    public CatalogEventConsumer(FavouriteUserBsProperties favouriteUserBsConfigurationProperties,
                                FavouriteUserItemService favouriteUserItemService) {
        MessagingCatalogBsProperties messagingCatalogBsProperties = favouriteUserBsConfigurationProperties.messaging().getCatalogBs();
        addHandler(messagingCatalogBsProperties.getResourceDeletedEventType(), new TypeReference<>() {
        }, this::handleResourceDeleted);
        addHandler(messagingCatalogBsProperties.getResourceStatusChangeEventType(), new TypeReference<>() {
                },
                this::handleResourceStatusChange);

        this.favouriteUserItemService = favouriteUserItemService;
    }

    private void handleResourceDeleted(UntypedMessagingEventDto eventPayload, CloudEvent cloudEvent) {
        favouriteUserItemService.deleteByContextAndItemId(ItemContext.CATALOG, null, eventPayload.getId(),
                eventPayload.getUser());
    }

    private void handleResourceStatusChange(MessagingEventDto<CatalogItemMessagingStatusAdditionalDataDto> eventPayload, CloudEvent cloudEvent) {
        CatalogItemMessagingStatusAdditionalDataDto data = eventPayload.getAdditionalData();

        // when the item creation is approved for the first time
        // generate the item created by favourite user event
        if (STATUS_APPROVED.equalsIgnoreCase(eventPayload.getStatus())
                && OPERATION_TYPE_CREATE.equalsIgnoreCase(data.getOperationType())
                && data.getMaintainerId() != null) {

            favouriteUserItemService.publishResourceCreatedByFolloweeEvent(ItemContext.CATALOG.getId(),
                    data.getMaintainerId().toString(), eventPayload);
        }
    }

}