package com.finconsgroup.itserr.marketplace.favourite.user.bs.service;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2BusinessException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2DuplicateResourceException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.FavouriteUserItemDetail;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.InputCreateFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.InputFindFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.InputPatchFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.OutputFavouriteUserItemDetailDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.OutputFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.enums.ItemContext;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;
import com.finconsgroup.itserr.messaging.dto.MessagingEventUserDto;
import org.springframework.lang.NonNull;

import java.util.List;
import java.util.UUID;

/**
 * Service for handling business logic related to favourite user items.
 */
public interface FavouriteUserItemService {
    /**
     * Creates a new favourite user item.
     *
     * @param dto Creation request. Cannot be null.
     * @return Created favourite user item. Never null.
     * @throws WP2DuplicateResourceException if the favourite user item already exists
     */
    @NonNull
    OutputFavouriteUserItemDto create(@NonNull InputCreateFavouriteUserItemDto dto);

    /**
     * Retrieves a favourite user item by id.
     *
     * @param favouriteUserItemId the id of the favourite user item to retrieve
     * @param <T> the type of the detail object
     * @return an {@link OutputFavouriteUserItemDto} representing the favourite user item details
     * @throws WP2ResourceNotFoundException if no favourite user item is found with the given id
     */
    @NonNull
    <T extends FavouriteUserItemDetail> OutputFavouriteUserItemDetailDto<T> findById(@NonNull UUID favouriteUserItemId);

    /**
     * Deletes the favourite user item by id.
     *
     * @param favouriteUserItemId the id of the favourite user item to delete
     * @throws WP2BusinessException         if delete is not allowed by configuration
     * @throws WP2ResourceNotFoundException if no favourite user item is found with the given id
     */
    void deleteById(@NonNull UUID favouriteUserItemId);

    /**
     * Updates only the provided fields of the favourite user item.
     *
     * @param favouriteUserItemId the id of the favourite user item to update
     * @param dto Patch request. Cannot be null.
     * @throws WP2ResourceNotFoundException if no favourite user item is found with the given id
     */
    @NonNull
    OutputFavouriteUserItemDto patchById(@NonNull UUID favouriteUserItemId, @NonNull InputPatchFavouriteUserItemDto dto);

    /**
     * Retrieves a page of results of matching favourite user item for the given context.
     *
     * @param dto  the dto containing input details
     * @param <T> the type of the detail object
     * @return a page of {@link OutputFavouriteUserItemDto} representing the matching favourite user items
     */
    @NonNull
    <T extends FavouriteUserItemDetail> OutputPageDto<OutputFavouriteUserItemDetailDto<T>> findByContext(@NonNull InputFindFavouriteUserItemDto dto);

    /**
     * Retrieves a list of results of matching favourite user item for the given context and item ids.
     *
     * @param context  the context to match
     * @param subContext the sub context to match
     * @param itemIds the item ids to match
     * @param itemIdSeparator the separator to split item ids
     * @return a list of {@link OutputFavouriteUserItemDto} representing the matching favourite user items
     */
    @NonNull
    List<OutputFavouriteUserItemDto> findByContextAndItemIds(@NonNull ItemContext context,
                                                             String subContext,
                                                             @NonNull String itemIds,
                                                             String itemIdSeparator);

    /**
     * Deletes the favourite user item for all users by item id.
     *
     * @param context  the context to match
     * @param subContext the sub context to match
     * @param itemId the id of the favourite user item to delete
     * @param messagingEventUserDto the user who deleted the source item.
     */
    void deleteByContextAndItemId(@NonNull ItemContext context,
                                  String subContext,
                                  @NonNull String itemId,
                                  MessagingEventUserDto messagingEventUserDto);

    /**
     * Publish an event to notify all the followers that a resource was created by the followee.
     *
     * @param context the context indicating the type of resource
     * @param followeeRef the followee identifier could be id, username or email
     * @param resourceEventDto the messaging event dto for the resource created event
     */
    void publishResourceCreatedByFolloweeEvent(@NonNull String context,
                                               @NonNull String followeeRef,
                                               @NonNull MessagingEventDto<?> resourceEventDto);
}
