package com.finconsgroup.itserr.marketplace.discussion.dm.entity;

import com.finconsgroup.itserr.marketplace.discussion.dm.enums.ReactionType;
import jakarta.persistence.*;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

import java.time.Instant;
import java.util.UUID;

@Entity
@Table(name = "discussion_reaction")
@NoArgsConstructor
@AllArgsConstructor
@Data
@Builder
public class DiscussionReaction {
    /**
     * Represents the universally unique identifier (UUID) for the DiscussionReaction entity.
     * This value is automatically generated using the UUID generation strategy.
     * It serves as the primary key and unique identifier for the entity in the database.
     */
    @Id
    @GeneratedValue(strategy = GenerationType.UUID)
    private UUID id;

    /**
     * Represents the association between a {@code DiscussionReaction} entity and a {@code Discussion} entity.
     * This field establishes a many-to-one relationship, where multiple {@code DiscussionReaction} entities
     * can be linked to a single {@code Discussion}.
     *
     * The association is configured with:
     * - {@code FetchType.LAZY} to delay the fetching of the related {@code Discussion} entity until accessed.
     * - {@code @JoinColumn} annotation to specify the foreign key column in the database, which is named
     *   "discussion_id" and marked as non-nullable, ensuring that every {@code DiscussionReaction} must be
     *   linked to a valid {@code Discussion}.
     */
    @ManyToOne(fetch = FetchType.LAZY)
    @JoinColumn(name = "discussion_id", nullable = false)
    private Discussion discussion;

    /**
     * Represents the unique identifier of a user who interacted with the discussion.
     * This field is used to associate a reaction with a specific user in the system.
     * It helps in tracking which user performed a given reaction (e.g., like, dislike) on a discussion entity.
     */
    private String userId;
    /**
     * Represents the type of reaction associated with a discussion.
     * This variable stores the enumerated type of reaction (LIKE, CELEBRATE, LOVE, INSIGHTFUL, CURIOUS)
     * that allows users to express their feedback or sentiment
     * regarding a particular discussion entity.
     */
    @Enumerated(EnumType.STRING)
    private ReactionType reaction;
    /**
     * The date and time when the reaction was created.
     * This field captures the timestamp for the creation of a
     * DiscussionReaction entity and is automatically initialized
     * with the current time when the entity is persisted in the database.
     */
    private Instant createdAt;

    /**
     * Callback method annotated with {@code @PrePersist} that is invoked
     * automatically by the persistence context before the entity is persisted to the database.
     * This method sets the {@code createdAt} field to the current date and time,
     * ensuring that the creation timestamp is recorded for the entity.
     */
    @PrePersist
    public void onCreate() {
        createdAt = Instant.now();
    }
}
