package org.finconsgroup.itserr.criterion.security.config;

import io.github.resilience4j.circuitbreaker.CircuitBreaker;
import io.github.resilience4j.circuitbreaker.CircuitBreakerConfig;
import io.github.resilience4j.circuitbreaker.CircuitBreakerRegistry;
import io.github.resilience4j.retry.Retry;
import io.github.resilience4j.retry.RetryConfig;
import io.github.resilience4j.retry.RetryRegistry;
import lombok.extern.slf4j.Slf4j;
import org.finconsgroup.itserr.criterion.security.aspect.ExternalAuthAspect;
import org.finconsgroup.itserr.criterion.security.service.ExternalAuthService;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.context.annotation.Bean;
import org.springframework.http.client.SimpleClientHttpRequestFactory;
import org.springframework.web.client.ResourceAccessException;
import org.springframework.web.client.RestTemplate;

import java.io.IOException;
import java.net.ConnectException;
import java.net.SocketTimeoutException;
import java.util.concurrent.TimeoutException;

/**
 * Autoconfiguration for the external authorization module.
 * Automatically activated when the library is on the classpath.
 */
@AutoConfiguration
@EnableConfigurationProperties(ExternalAuthProperties.class)
@ConditionalOnProperty(prefix = "wp3.external-auth", name = "enabled", havingValue = "true", matchIfMissing = true)
@Slf4j
public class ExternalAuthAutoConfiguration {

    public static final String CIRCUIT_BREAKER_NAME = "externalAuth";
    public static final String RETRY_NAME = "externalAuth";

    /**
     * Create RestTemplate with configured timeouts.
     * Uses ObjectProvider to avoid IDE warnings in library projects.
     */
    @Bean
    @ConditionalOnMissingBean(name = "externalAuthRestTemplate")
    public RestTemplate externalAuthRestTemplate(ObjectProvider<RestTemplateBuilder> restTemplateBuilderProvider,
                                                 ExternalAuthProperties properties) {
        log.debug("Creating RestTemplate with connectTimeout={}, readTimeout={}",
                properties.getConnectTimeout(), properties.getReadTimeout());

        RestTemplateBuilder builder = restTemplateBuilderProvider.getIfAvailable();

        if (builder != null) {
            return builder
                    .connectTimeout(properties.getConnectTimeout())
                    .readTimeout(properties.getReadTimeout())
                    .build();
        }

        // Fallback if RestTemplateBuilder is not available
        SimpleClientHttpRequestFactory factory = new SimpleClientHttpRequestFactory();
        factory.setConnectTimeout((int) properties.getConnectTimeout().toMillis());
        factory.setReadTimeout((int) properties.getReadTimeout().toMillis());
        return new RestTemplate(factory);
    }

    @Bean
    @ConditionalOnMissingBean
    public CircuitBreakerRegistry externalAuthCircuitBreakerRegistry(ExternalAuthProperties properties) {
        var cbProps = properties.getCircuitBreaker();

        CircuitBreakerConfig config = CircuitBreakerConfig.custom()
                .failureRateThreshold(cbProps.getFailureRateThreshold())
                .waitDurationInOpenState(cbProps.getWaitDurationInOpenState())
                .permittedNumberOfCallsInHalfOpenState(cbProps.getPermittedNumberOfCallsInHalfOpenState())
                .slidingWindowSize(cbProps.getSlidingWindowSize())
                .minimumNumberOfCalls(cbProps.getMinimumNumberOfCalls())
                .recordExceptions(
                        ResourceAccessException.class,
                        ConnectException.class,
                        SocketTimeoutException.class,
                        TimeoutException.class,
                        IOException.class
                )
                .build();

        return CircuitBreakerRegistry.of(config);
    }

    @Bean
    @ConditionalOnMissingBean
    public RetryRegistry externalAuthRetryRegistry(ExternalAuthProperties properties) {
        var retryProps = properties.getRetry();

        RetryConfig config = RetryConfig.custom()
                .maxAttempts(retryProps.getMaxAttempts())
                .waitDuration(retryProps.getWaitDuration())
                .intervalFunction(attempt ->
                        Math.min(
                                (long) (retryProps.getWaitDuration().toMillis() *
                                        Math.pow(retryProps.getMultiplier(), attempt - 1)),
                                retryProps.getMaxWaitDuration().toMillis()
                        )
                )
                .retryExceptions(
                        ResourceAccessException.class,
                        ConnectException.class,
                        SocketTimeoutException.class,
                        TimeoutException.class,
                        IOException.class
                )
                .build();

        return RetryRegistry.of(config);
    }

    @Bean
    public CircuitBreaker externalAuthCircuitBreaker(CircuitBreakerRegistry registry) {
        CircuitBreaker circuitBreaker = registry.circuitBreaker(CIRCUIT_BREAKER_NAME);

        circuitBreaker.getEventPublisher()
                .onStateTransition(event ->
                        log.warn("Circuit Breaker '{}' state changed: {} -> {}",
                                CIRCUIT_BREAKER_NAME,
                                event.getStateTransition().getFromState(),
                                event.getStateTransition().getToState()))
                .onFailureRateExceeded(event ->
                        log.warn("Circuit Breaker '{}' failure rate exceeded: {}%",
                                CIRCUIT_BREAKER_NAME,
                                event.getFailureRate()))
                .onCallNotPermitted(event ->
                        log.warn("Circuit Breaker '{}' rejected call - circuit is OPEN",
                                CIRCUIT_BREAKER_NAME));

        return circuitBreaker;
    }

    @Bean
    public Retry externalAuthRetry(RetryRegistry registry) {
        Retry retry = registry.retry(RETRY_NAME);

        retry.getEventPublisher()
                .onRetry(event ->
                        log.warn("Retry '{}' attempt #{} failed, waiting before next attempt...",
                                RETRY_NAME,
                                event.getNumberOfRetryAttempts()))
                .onError(event ->
                        log.error("Retry '{}' exhausted after {} attempts",
                                RETRY_NAME,
                                event.getNumberOfRetryAttempts()));

        return retry;
    }


    @Bean
    @ConditionalOnMissingBean
    public ExternalAuthService externalAuthService(
            @Qualifier("externalAuthRestTemplate") RestTemplate externalAuthRestTemplate,
            ExternalAuthProperties properties,
            CircuitBreaker externalAuthCircuitBreaker,
            Retry externalAuthRetry) {
        return new ExternalAuthService(
                externalAuthRestTemplate,
                properties,
                externalAuthCircuitBreaker,
                externalAuthRetry
        );
    }

    @Bean
    @ConditionalOnMissingBean
    public ExternalAuthAspect externalAuthAspect(ExternalAuthService externalAuthService,
                                                 ExternalAuthProperties properties) {
        log.info("External Auth ENABLED - URL: {}, FailOpen: {}",
                properties.getUrl(),
                properties.isFailOpen());
        return new ExternalAuthAspect(externalAuthService, properties);
    }
}