/*******************************************************************************
 * Copyright (c) 2013 Metadata Technology Ltd.
 *  
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the GNU Lesser General Public License v 3.0 
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl.html
 * 
 * This file is part of the SDMX Component Library.
 * 
 * The SDMX Component Library is free software: you can redistribute it and/or 
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 * 
 * The SDMX Component Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser 
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with The SDMX Component Library If not, see 
 * http://www.gnu.org/licenses/lgpl.
 * 
 * Contributors:
 * Metadata Technology - initial API and implementation
 ******************************************************************************/
package org.sdmxsource.sdmx.ediparser.manager.impl;

import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;

import org.sdmxsource.sdmx.api.constants.ExceptionCode;
import org.sdmxsource.sdmx.api.constants.SDMX_STRUCTURE_TYPE;
import org.sdmxsource.sdmx.api.exception.SdmxNotImplementedException;
import org.sdmxsource.sdmx.api.model.beans.SdmxBeans;
import org.sdmxsource.sdmx.api.model.beans.base.MaintainableBean;
import org.sdmxsource.sdmx.api.util.ReadableDataLocation;
import org.sdmxsource.sdmx.ediparser.engine.writer.impl.EDIStructureWriterEngineImpl;
import org.sdmxsource.sdmx.ediparser.manager.EdiParseManager;
import org.sdmxsource.sdmx.ediparser.model.EDIWorkspace;
import org.sdmxsource.sdmx.ediparser.model.impl.EDIWorkspaceImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;


@Service
public class EdiParseManagerImpl implements EdiParseManager {
	
	@Autowired
	private EDIStructureWriterEngineImpl ediStructureEngine;
	
	@Override
	public EDIWorkspace parseEDIMessage(ReadableDataLocation ediMessageLocation) {
		return new EDIWorkspaceImpl(ediMessageLocation);
	}

	@Override
	public void writeToEDI(SdmxBeans beans, OutputStream out) {
		//Check that what is being written is supported by EDI
		validateSupport(beans);
		
		ediStructureEngine.writeToEDI(beans, out);
	}
	
	/**
	 * Validates all the Maintainable Artefacts in the beans container are supported by the SDMX v1.0 syntax 
	 * @param beans
	 */
	private void validateSupport(SdmxBeans beans) {
		List<SDMX_STRUCTURE_TYPE> supportedStructres = new ArrayList<SDMX_STRUCTURE_TYPE>();
		supportedStructres.add(SDMX_STRUCTURE_TYPE.AGENCY_SCHEME);
		supportedStructres.add(SDMX_STRUCTURE_TYPE.DSD);
		supportedStructres.add(SDMX_STRUCTURE_TYPE.CONCEPT_SCHEME);
		supportedStructres.add(SDMX_STRUCTURE_TYPE.CODE_LIST);
		
		for(MaintainableBean maintainableBean : beans.getAllMaintainables()) {
			if(!supportedStructres.contains(maintainableBean.getStructureType())) {
				throw new SdmxNotImplementedException(ExceptionCode.UNSUPPORTED, maintainableBean.getStructureType().getType() + " is not a supported by SDMX-EDI");
			}
		}
	}
}
